package service

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"
	"strconv"

	"a.yandex-team.ru/security/yadi/yadi/internal/config"
	"a.yandex-team.ru/security/yadi/yadi/internal/httputils"
	"a.yandex-team.ru/security/yadi/yadi/internal/results"
)

type YadiResponse struct {
	Ok     bool        `json:"ok"`
	Result interface{} `json:"result"`
	Error  string      `json:"error"`
}

var defaultUserAgent string

func init() {
	defaultUserAgent = fmt.Sprintf("Yadi/v%s", config.Version)
}

func ValidateAbcService(service string) error {
	params := url.Values{
		"slug": {service},
	}

	uri := fmt.Sprintf("%s/api/v1/common/abc?%s",
		config.YadiHost, params.Encode())

	req, _ := http.NewRequest("GET", uri, nil)
	err := makeRequest(req, "", nil)
	if err != nil {
		return err
	}
	return nil
}

func ValidateProject(service, name string) error {
	params := url.Values{
		"service": {service},
		"name":    {name},
	}

	uri := fmt.Sprintf("%s/api/v1/common/project?%s",
		config.YadiHost, params.Encode())

	req, _ := http.NewRequest("GET", uri, nil)
	err := makeRequest(req, "", nil)
	if err != nil {
		return err
	}
	return nil
}

func ValidateProjectActive(service, name string) (bool, error) {
	params := url.Values{
		"service": {service},
		"name":    {name},
	}

	uri := fmt.Sprintf("%s/api/v1/common/project/activate/?%s",
		config.YadiHost, params.Encode())

	var result struct {
		Result bool
	}

	req, _ := http.NewRequest("GET", uri, nil)
	err := makeRequest(req, "", &result)
	if err != nil {
		return false, err
	}

	return result.Result, nil
}

func NewProject(project *ProjectInfo) (*NewProjectInfo, error) {
	uri := fmt.Sprintf("%s/api/v1/project/new", config.YadiHost)
	body, err := json.Marshal(project)
	if err != nil {
		return nil, err
	}

	resp, err := httputils.HTTPClient.Post(uri, "application/json", bytes.NewBuffer(body))
	if err != nil {
		return nil, fmt.Errorf("failed to validate project: %s", err.Error())
	}
	defer httputils.GracefulClose(resp.Body)

	var result struct {
		Ok     bool
		Result *NewProjectInfo
		Error  *string
	}
	err = json.NewDecoder(resp.Body).Decode(&result)
	if err == nil && !result.Ok {
		return nil, errors.New(*result.Error)
	}

	if resp.StatusCode != 200 {
		return nil, errors.New("unknown")
	}

	return result.Result, nil
}

func GetProject(project *ProjectInfo) (*ProjectInfo, error) {
	uri := fmt.Sprintf("%s/api/v1/project/%s/%s/",
		config.YadiHost, url.PathEscape(project.Service), url.PathEscape(project.Name))

	req, err := http.NewRequest("GET", uri, nil)
	if err != nil {
		return nil, err
	}

	var result struct {
		Result *ProjectInfo
	}

	err = makeRequest(req, project.AccessToken, &result)
	if err != nil {
		return nil, fmt.Errorf("failed to get project info: %s", err.Error())
	}

	return result.Result, nil
}

func SyncProject(syncInfo *ProjectSyncInfo) error {
	uri := fmt.Sprintf("%s/api/v1/project/%s/%s/sync",
		config.YadiHost, url.PathEscape(syncInfo.Service), url.PathEscape(syncInfo.Name))
	body, err := json.Marshal(syncInfo)
	if err != nil {
		return err
	}

	req, err := http.NewRequest("POST", uri, bytes.NewBuffer(body))
	if err != nil {
		return err
	}

	err = makeRequest(req, syncInfo.AccessToken, nil)
	if err != nil {
		return fmt.Errorf("failed to sync project: %s", err.Error())
	}

	return nil
}

func GetProjectDependencies(project *MinimumProjectInfo, depth int) ([]results.List, error) {
	params := url.Values{
		"depth": {strconv.Itoa(depth)},
	}

	uri := fmt.Sprintf("%s/api/v1/project/%s/%s/dependencies?%s",
		config.YadiHost, url.PathEscape(project.Service), url.PathEscape(project.Name), params.Encode())

	req, err := http.NewRequest("GET", uri, nil)
	if err != nil {
		return nil, err
	}
	var result struct {
		Result []results.List
	}

	err = makeRequest(req, project.AccessToken, &result)
	if err != nil {
		return nil, fmt.Errorf("failed to get project dependencies: %s", err.Error())
	}
	return result.Result, nil
}

func makeRequest(request *http.Request, accessToken string, result interface{}) error {
	if accessToken != "" {
		request.Header.Set("Authorization", fmt.Sprintf("OAuth %s", accessToken))
	}
	request.Header.Set("User-Agent", defaultUserAgent)
	request.Header.Set("Content-Type", "application/json")

	resp, err := httputils.HTTPClient.Do(request)
	if err != nil {
		return err
	}
	defer httputils.GracefulClose(resp.Body)

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return err
	}

	if resp.StatusCode != 200 {
		var res YadiResponse
		err = json.Unmarshal(body, &res)
		if err == nil && res.Error != "" {
			return errors.New(res.Error)
		}
		return fmt.Errorf("yadi return not 200 status code (%d), response: %s", resp.StatusCode, string(body))
	}

	if result != nil {
		err = json.Unmarshal(body, result)
		if err != nil {
			return err
		}
	}

	return nil
}
