package analyze

import (
	"context"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/cvs"
	"a.yandex-team.ru/security/yadi/yadi/pkg/feed"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager"
)

var ErrCanceled = xerrors.New("canceled")

type (
	analyzer struct {
		feedFetcher     *feed.Fetcher
		checkRootModule bool
		trackStats      bool
		suggestUpdate   bool
		vulnsExclude    map[string]struct{}
	}
)

func NewAnalyzer(opts ...AnalyzerOption) *analyzer {
	analyzer := &analyzer{
		feedFetcher: feed.New(feed.Options{
			MinimumSeverity: cvs.MediumSeverity,
			FeedURI:         feed.DefaultURI,
		}),
		trackStats:    false,
		suggestUpdate: false,
	}

	for _, opt := range opts {
		opt(analyzer)
	}

	return analyzer
}

func (a *analyzer) Analyze(ctx context.Context, req Request) (ResultAnalyze, error) {
	vulnerabilities, err := a.feedFetcher.Fetch(req.PackageManager.Language())
	if err != nil {
		return ResultAnalyze{}, xerrors.Errorf("failed to fetch vulnerabilities: %w", err)
	}

	modules, err := req.PackageManager.RootModules()
	if err != nil {
		return ResultAnalyze{}, xerrors.Errorf("failed to get root modules: %w", err)
	}

	if len(modules) == 0 {
		return ResultAnalyze{}, nil
	}

	walker := issueWalker{
		ctx:           ctx,
		suggestUpdate: a.suggestUpdate && req.PackageManager.CanSuggest(),
		pm:            req.PackageManager,
		vulns:         vulnerabilities,
		vulnsExclude:  a.vulnsExclude,
	}

	if !walker.suggestUpdate && walker.pm.Cacheable() {
		// we use cached issues only if we can't suggest module updates
		walker.issueCache = newIssueCache()
	}

	results := ResultAnalyze{
		Issues: make(map[string]IssueList, len(modules)),
		Stats:  make(map[string]DependencyStats, len(modules)),
	}

	for _, module := range modules {
		if a.trackStats {
			walker.stats = &DependencyStats{}
		}

		issues, err := walker.Walk(module)
		if err != nil {
			simplelog.Error("failed to walk module tree", "module", module.Name, "local_path", module.LocalPath)
			continue
		}

		results.Issues[module.LocalPath] = issues
		if a.trackStats {
			results.Stats[module.LocalPath] = *walker.stats
		}
	}

	return results, err
}

func (a *analyzer) AnalyzePkg(ctx context.Context, req PkgRequest) (ResultAnalyzePkg, error) {
	module, err := req.PackageManager.ResolveRemoteDependency(
		manager.Dependency{
			Name:        req.PackageName,
			RawVersions: req.PackageVersion,
			Language:    req.PackageManager.Language(),
		},
		manager.ZeroModule,
	)
	if err != nil {
		return nil, ErrPkgNotFound.Wrap(err)
	}

	vulnerabilities, err := a.feedFetcher.Fetch(req.PackageManager.Language())
	if err != nil {
		return nil, xerrors.Errorf("failed to fetch vulnerabilities: %w", err)
	}

	walker := issueWalker{
		ctx:             ctx,
		suggestUpdate:   a.suggestUpdate && req.PackageManager.CanSuggest(),
		checkRootModule: true,
		pm:              req.PackageManager,
		vulns:           vulnerabilities,
		vulnsExclude:    a.vulnsExclude,
	}

	if !walker.suggestUpdate && walker.pm.Cacheable() {
		// we use cached issues only if we can't suggest module updates
		walker.issueCache = newIssueCache()
	}

	return walker.Walk(module)
}

func (a *analyzer) Walk(ctx context.Context, req WalkRequest) error {
	if req.Consumer == nil {
		return xerrors.New("consumer must be provided")
	}

	modules, err := req.PackageManager.RootModules()
	if err != nil {
		return xerrors.Errorf("failed to get root modules: %w", err)
	}

	if len(modules) == 0 {
		return nil
	}

	walker := listWalker{
		ctx:      ctx,
		pm:       req.PackageManager,
		consumer: req.Consumer,
	}

	for _, module := range modules {
		if a.trackStats {
			walker.stats = &DependencyStats{}
		}

		err := walker.Walk(module)
		if err != nil {
			simplelog.Error("failed to walk module tree", "module", module.Name, "local_path", module.LocalPath)
			continue
		}
	}

	return nil
}

// TODO(buglloc): use consumer
func (a *analyzer) ListPkg(ctx context.Context, req ListPkgRequest) (result ResultListPkg, resultErr error) {
	module, err := req.PackageManager.ResolveRemoteDependency(
		manager.Dependency{
			Name:        req.PackageName,
			RawVersions: req.PackageVersion,
			Language:    req.PackageManager.Language(),
		},
		manager.ZeroModule,
	)
	if err != nil {
		resultErr = ErrPkgNotFound.Wrap(err)
		return
	}

	workCtx := workContext{
		pm: req.PackageManager,
	}

	modulesTree, err := a.resolveModuleTree(ctx, workCtx, module, "", nil)
	if err != nil {
		resultErr = xerrors.Errorf("failed to list modules: %w", err)
		return
	}

	result = modulesTree.ResultModuleTree()
	return
}

// TODO(buglloc): drop me
func (a *analyzer) resolveModuleTree(ctx context.Context, wCtx workContext, module manager.Module, from string, parents []manager.Module) (moduleTree, error) {
	result := moduleTree{
		Module: module,
		From:   from,
	}

	if isCircularDep(module, parents) {
		simplelog.Debug("circular recursion",
			"module", module.String(),
			"path", manager.BuildTextPath(parents),
		)
		return result, nil
	}

	select {
	case <-ctx.Done():
		return moduleTree{}, ErrCanceled
	default:
	}

	newParents := append(parents, module)
	for _, packageDep := range module.Dependencies {
		if _, skip := wCtx.depExcludes[packageDep.Name]; skip {
			continue
		}

		if wCtx.stats != nil {
			wCtx.stats.Increment(packageDep.IsDev)
		}

		subModule, err := searchDependencyModule(wCtx.pm, module, packageDep)
		if err != nil {
			continue
		}

		subDependencies, err := a.resolveModuleTree(ctx, wCtx, subModule, packageDep.FullName(), newParents)
		if err != nil {
			return result, err
		}

		result.Dependencies = append(result.Dependencies, subDependencies)
	}

	return result, nil
}
