package analyze

import (
	"context"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/yadi/pkg/feed"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager"
)

type issueWalker struct {
	ctx             context.Context
	suggestUpdate   bool
	checkRootModule bool
	pm              manager.PackageManager
	stats           *DependencyStats
	issues          IssueList
	vulns           feed.Vulnerabilities
	vulnsExclude    map[string]struct{}
	issueCache      *issueCache
}

func (w *issueWalker) Walk(module manager.Module) (IssueList, error) {
	simplelog.Info("walk thought module tree", "name", module.Name, "path", module.LocalPath)
	issues, err := w.walkModule(module, nil)
	if err != nil {
		return nil, err
	}

	simplelog.Info("done")
	return issues, nil
}

func (w *issueWalker) walkModule(module manager.Module, parents []manager.Module) (IssueList, error) {
	select {
	case <-w.ctx.Done():
		return nil, ErrCanceled
	default:
	}

	if w.issueCache != nil {
		issues, ok := w.issueCache.GetFor(module)
		if ok {
			return issues, nil
		}
	}

	if isCircularDep(module, parents) {
		simplelog.Debug("circular recursion",
			"module", module.String(),
			"path", manager.BuildTextPath(parents),
		)
		return nil, nil
	}

	var issues IssueList
	topLevel := len(parents) == 0
	newParents := append(parents, module)

	// check pkg dependencies for vulns
	for _, packageDep := range module.Dependencies {
		if w.stats != nil {
			w.stats.Increment(packageDep.IsDev)
		}

		subModule, err := searchDependencyModule(w.pm, module, packageDep)
		if err != nil {
			continue
		}

		subIssues, err := w.walkModule(subModule, newParents)
		if err != nil {
			return nil, err
		}

		if len(subIssues) == 0 {
			continue
		}

		if topLevel {
			issues = append(issues, subIssues...)
			continue
		}

		// TODO(buglloc): do something better
		fixedIssues := make(IssueList, len(subIssues))
		copy(fixedIssues, subIssues)
		for i := range fixedIssues {
			fixedIssues[i].Path = append([]string{module.String()}, fixedIssues[i].Path...)
		}
		issues = append(issues, fixedIssues...)
	}

	defer func() {
		if w.issueCache != nil {
			w.issueCache.SaveFor(module, issues)
		}
	}()

	if topLevel && !w.checkRootModule {
		return issues, nil
	}

	// check pkg vulns
	for _, vuln := range w.vulns.ForPackage(module.Name) {
		if _, skip := w.vulnsExclude[vuln.ID]; skip {
			continue
		}

		if !vuln.CheckVersion(module.Version) {
			continue
		}

		var suggest []string
		if w.suggestUpdate && vuln.PatchExists {
			suggest = w.pm.SuggestModuleUpdate(module, vuln.Versions, parents)
		}

		issue := Issue{
			Vulnerability: vuln,
			Version:       module.Version.String(),
			Suggest:       suggest,
			Suggestable:   w.suggestUpdate,
			Path:          []string{module.String()},
		}

		issues = append(issues, issue)
	}

	return issues, nil
}
