package analyze

import (
	"context"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager"
)

type listWalker struct {
	ctx      context.Context
	pm       manager.PackageManager
	stats    *DependencyStats
	consumer Consumer
}

func (w *listWalker) Walk(module manager.Module) error {
	w.consumer.EnterRootModule(module, w.pm)
	defer w.consumer.ExitRootModule(w.stats)

	simplelog.Info("walk thought module tree", "name", module.Name, "path", module.LocalPath)
	err := w.walkModule(module, "", nil)
	if err != nil {
		return err
	}

	simplelog.Info("done")
	return nil
}

func (w *listWalker) walkModule(module manager.Module, from string, parents []manager.Module) error {
	if len(parents) > 0 {
		// not a root module
		w.consumer.EnterModule(module, from, parents)
		defer w.consumer.ExitModule()
	}

	if isCircularDep(module, parents) {
		simplelog.Debug("circular recursion",
			"module", module.String(),
			"path", manager.BuildTextPath(parents),
		)
		return nil
	}

	select {
	case <-w.ctx.Done():
		return ErrCanceled
	default:
	}

	for _, packageDep := range module.Dependencies {
		if w.stats != nil {
			w.stats.Increment(packageDep.IsDev)
		}

		subModule, err := searchDependencyModule(w.pm, module, packageDep)
		if err != nil {
			continue
		}

		err = w.walkModule(subModule, packageDep.FullName(), append(parents, module))
		if err != nil {
			return err
		}
	}

	return nil
}

func searchDependencyModule(pm manager.PackageManager, parent manager.Module, dep manager.Dependency) (manager.Module, error) {
	if pm.CanLocal() {
		result, err := pm.ResolveLocalDependency(dep, parent)
		if err == nil {
			return result, nil
		}
		simplelog.Debug("failed to resolve local dependency", "dependency", dep.FullName(), "err", err)
	}

	if pm.CanRemote() {
		result, err := pm.ResolveRemoteDependency(dep, parent)
		if err == nil {
			return result, nil
		}
		simplelog.Debug("failed to resolve remote dependency", "dependency", dep.FullName(), "err", err)
	}

	return manager.ZeroModule, xerrors.New("failed to find dependency")
}

func isCircularDep(module manager.Module, parents []manager.Module) bool {
	for i := len(parents) - 1; i >= 0; i-- {
		if parents[i].Name == module.Name {
			return true
		}
	}
	return false
}
