package analyze

//go:generate easyjson

import (
	"context"
	"fmt"
	"sync/atomic"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/yadi/libs/cvs"
	"a.yandex-team.ru/security/yadi/yadi/pkg/feed"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager"
)

var (
	ErrPkgNotFound = xerrors.NewSentinel("pkg not found")
)

type Analyzer interface {
	Analyze(ctx context.Context, req Request) (result ResultAnalyze, err error)
	AnalyzePkg(ctx context.Context, req PkgRequest) (result ResultAnalyzePkg, err error)

	ListPkg(ctx context.Context, req ListPkgRequest) (result ResultListPkg, err error)

	Walk(ctx context.Context, req WalkRequest) error
}

type Consumer interface {
	EnterRootModule(module manager.Module, pm manager.PackageManager)
	EnterModule(module manager.Module, from string, parents []manager.Module)
	ExitModule()
	ExitRootModule(*DependencyStats)
}

type Request struct {
	// Target package manager
	PackageManager manager.PackageManager
}

type ContribRequest struct {
	// Target package manager
	PackageManager manager.PackageManager

	PackageName string

	PackageVersion string

	FullTree bool
}

type PkgRequest struct {
	// Target package manager
	PackageManager manager.PackageManager

	PackageName string

	PackageVersion string
}

type WalkRequest struct {
	// Target package manager
	PackageManager manager.PackageManager

	// ListConsumer that's will be notified
	Consumer Consumer
}

type ListPkgRequest struct {
	// Target package manager
	PackageManager manager.PackageManager

	PackageName string

	PackageVersion string

	// ListConsumer that's will be notified
	Consumer Consumer
}

//easyjson:json
type DependencyStats struct {
	total uint64
	dev   uint64
}

//easyjson:json
type ResultAnalyze struct {
	Issues map[string]IssueList
	Stats  map[string]DependencyStats
}

//easyjson:json
type ResultAnalyzeContrib struct {
	Issues     IssueList
	ModuleTree ModuleTree
}

type ResultAnalyzePkg = IssueList

//easyjson:json
type ResultList struct {
	Trees map[string]ModuleTree
	Stats map[string]DependencyStats
}

//easyjson:json
type ProjectInfo struct {
	ProjectName string       `json:"project_name"`
	Language    string       `json:"language"`
	Trees       []ModuleTree `json:"root_modules"`
}

type ResultListPkg = ModuleTree

//easyjson:json
type Issue struct {
	feed.Vulnerability
	Version     string   `json:"version"`
	Suggest     []string `json:"suggest"`
	Suggestable bool     `json:"-"`
	Path        []string `json:"path"`
}

//easyjson:json
type IssueList []Issue

//easyjson:json
type ModuleTree struct {
	PackageName  string       `json:"package_name"`
	Version      string       `json:"version"`
	Path         string       `json:"path"`
	License      string       `json:"license"`
	Language     string       `json:"language"`
	From         string       `json:"from"`
	Dependencies []ModuleTree `json:"dependencies"`
}

func (slice IssueList) Len() int {
	return len(slice)
}

func (slice IssueList) Less(i, j int) bool {
	return slice[i].CVSSScore > slice[j].CVSSScore
}

func (slice IssueList) Swap(i, j int) {
	slice[i], slice[j] = slice[j], slice[i]
}

func (y *Issue) Severity() string {
	return cvs.ToSeverity(y.CVSSScore)
}

func (y *Issue) String() string {
	return fmt.Sprintf("%s@%s", y.PackageName, y.Version)
}

func (s *DependencyStats) Stats() (total uint64, dev uint64) {
	total = atomic.LoadUint64(&s.total)
	dev = atomic.LoadUint64(&s.dev)
	return
}

func (s *DependencyStats) Increment(isDev bool) {
	atomic.AddUint64(&s.total, 1)
	if isDev {
		atomic.AddUint64(&s.dev, 1)
	}
}
