package feed

import (
	"bytes"
	"compress/gzip"
	"encoding/json"
	"errors"
	"io/ioutil"
	"path/filepath"
	"strings"
	"sync"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/versionarium"
	"a.yandex-team.ru/security/yadi/yadi/internal/httputils"
)

const (
	DefaultURI = "https://yadi.yandex-team.ru/db/{lang}.json.gz"
)

var (
	gzipMagic = []byte{0x1F, 0x8B, 0x08}
)

type (
	Fetcher struct {
		feedURI         string
		minimumSeverity float32
		cache           map[string]Vulnerabilities
		lock            sync.Mutex
	}

	Options struct {
		// Feed Url
		// Uri (or path to file) to download feed. May have placeholder for language
		// Example: https://yadi.yandex-team.ru/db/{lang}.json.gz
		FeedURI string

		// Ignore vulnerabilities that lower than that severity
		MinimumSeverity float32
	}
)

func New(opts Options) *Fetcher {
	result := &Fetcher{
		minimumSeverity: opts.MinimumSeverity,
		cache:           make(map[string]Vulnerabilities),
	}

	if opts.FeedURI != "" {
		result.feedURI = opts.FeedURI
	} else {
		result.feedURI = "https://yadi.yandex-team.ru/db/{lang}.json.gz"
	}

	return result
}

func (f *Fetcher) Fetch(language string) (Vulnerabilities, error) {
	f.lock.Lock()
	defer f.lock.Unlock()

	if result, ok := f.cache[language]; ok {
		return result, nil
	}

	result, err := f.fetchInternal(language)
	if err != nil {
		return nil, err
	}
	f.cache[language] = result
	return result, nil
}

func (f *Fetcher) CleanCache(language string) {
	f.lock.Lock()
	defer f.lock.Unlock()
	delete(f.cache, language)
}

func (f *Fetcher) fetchInternal(language string) (Vulnerabilities, error) {
	feedURI := FormatURI(f.feedURI, language)
	var raw []byte
	var err error
	if strings.HasPrefix(feedURI, "https://") || strings.HasPrefix(feedURI, "http://") {
		raw, err = fetchRemoteDB(feedURI)
	} else if filepath.IsAbs(feedURI) {
		raw, err = fetchLocalDB(feedURI)
	} else {
		err = errors.New("unsupported feed protocol: " + feedURI)
	}

	if err != nil {
		return nil, err
	}

	if len(raw) >= 3 && bytes.Equal(raw[:3], gzipMagic) {
		reader, err := gzip.NewReader(bytes.NewReader(raw))
		if err != nil {
			return nil, xerrors.Errorf("failed to create new gzip reader: %w", err)
		}

		raw, err = ioutil.ReadAll(reader)
		_ = reader.Close()
		if err != nil {
			return nil, xerrors.Errorf("failed to create read gzipped feed: %w", err)
		}
	}

	var feed Feed
	err = json.Unmarshal(raw, &feed)
	if err != nil {
		return nil, xerrors.Errorf("failed to parse feed: %w", err)
	}

	result := Vulnerabilities{}
	for _, entry := range feed {
		if entry.CVSSScore < f.minimumSeverity {
			continue
		}

		versions, err := versionarium.NewRange(language, entry.VulnerableVersions)
		if err != nil {
			simplelog.Error("failed to parse vulnerability versions: "+err.Error(),
				"module", entry.ModuleName, "versions", entry.VulnerableVersions)
			continue
		}

		result[entry.ModuleName] = append(
			result[entry.ModuleName],
			Vulnerability{
				ID:              entry.ID,
				PackageName:     entry.ModuleName,
				CVSSScore:       entry.CVSSScore,
				RawVersions:     entry.VulnerableVersions,
				PatchedVersions: entry.PatchedVersions,
				Versions:        versions,
				Summary:         entry.Summary,
				Reference:       entry.Reference,
				PatchExists:     entry.PatchExists,
			},
		)
	}

	return result, nil
}

func FormatURI(uri, language string) string {
	return strings.Replace(uri, "{lang}", language, -1)
}

func fetchRemoteDB(url string) ([]byte, error) {
	resp, err := httputils.DoGet(url)
	if err != nil {
		return nil, err
	}

	defer httputils.GracefulClose(resp.Body)
	if resp.StatusCode != 200 {
		return nil, xerrors.Errorf("unexpected feed status: %d", resp.StatusCode)
	}

	return ioutil.ReadAll(resp.Body)
}

func fetchLocalDB(path string) ([]byte, error) {
	return ioutil.ReadFile(path)
}
