package fsutils

import (
	"io"
	"os"
	"path/filepath"
)

func IsRegular(fi os.FileInfo) bool {
	return fi.Mode()&os.ModeType == 0
}

func IsFileExists(path string) bool {
	if stat, err := os.Stat(path); err == nil {
		return IsRegular(stat)
	}
	return false
}

func IsDirExists(path string) bool {
	if stat, err := os.Stat(path); err == nil {
		return stat.IsDir()
	}
	return false
}

func CopyFile(from, to string) error {
	src, err := os.Open(from)
	if err != nil {
		return err
	}
	defer func() { _ = src.Close() }()

	dst, err := os.Create(to)
	if err != nil {
		return err
	}
	defer func() { _ = dst.Close() }()
	_, err = io.Copy(dst, src)
	return err
}

func CopyTree(from, to string) error {
	return filepath.Walk(from, func(path string, info os.FileInfo, err error) error {
		if err != nil {
			return err
		}

		if path != from {
			path = path[len(from)+1:]
		} else {
			path = ""
		}

		if info.IsDir() {
			err := os.Mkdir(filepath.Join(to, path), 0755)
			if err != nil && !os.IsExist(err) {
				return err
			}
		} else if info.Mode()&os.ModeSymlink == 0 {
			return CopyFile(filepath.Join(from, path), filepath.Join(to, path))
		}

		return nil
	})
}

func CurrentDir() string {
	cwd, err := os.Getwd()
	if err != nil {
		return ""
	}
	return filepath.Base(cwd)
}
