package bazelmaven

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"path/filepath"
	"strings"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/versionarium"
	"a.yandex-team.ru/security/yadi/yadi/pkg/fsutils"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager"
)

const rootVersion = "0.0.0"

type (
	graph []Module

	resolveOpts struct {
		targetPath string
		withDev    bool
	}

	coordinates struct {
		raw, groupID, artifactID, packaging, version, classifier string
	}

	Module struct {
		coordinates
		dependencies []string
	}

	mavenInstallJSON struct {
		Tree struct {
			Dependencies []Module `json:"dependencies"`
		} `json:"dependency_tree"`
	}
)

func (m *Module) UnmarshalJSON(data []byte) error {
	var rawmod struct {
		Raw          string   `json:"coord"`
		Dependencies []string `json:"directDependencies"`
	}
	if err := json.Unmarshal(data, &rawmod); err != nil {
		return err
	}
	parsed, err := parseDependencyName(rawmod.Raw)
	if err != nil {
		return err
	}
	m.coordinates = *parsed
	m.dependencies = rawmod.Dependencies
	return nil
}

func (m Module) dependsOn(dst Module) bool {
	for _, name := range m.dependencies {
		if dst.raw == name {
			return true
		}
	}
	return false
}

func (m Module) isRootDependency(g graph) bool {
	for _, node := range g {
		if node.dependsOn(m) {
			return false
		}
	}
	return true
}

func (m Module) IsDev() bool {
	return false
}

func (n coordinates) HasClassifierSpecification() bool {
	return n.classifier != ""
}

func (n coordinates) Version() string {
	return n.version
}

func (n coordinates) Name() string {
	return fmt.Sprintf("%s:%s", n.groupID, n.artifactID)
}

func (g graph) resolveTree(opts resolveOpts) (manager.Module, error) {

	moduleName := filepath.Base(filepath.Dir(opts.targetPath))
	moduleVersion := versionarium.MustNewVersion(language, rootVersion)

	rootDeps := []manager.Dependency{}
	for _, node := range g {
		if node.isRootDependency(g) && !node.HasClassifierSpecification() {
			dep, err := g.resolve(node, opts.withDev)
			if err != nil {
				simplelog.Error("failed to resolve dependency", "name", node.raw, "err", err)
				continue
			}
			rootDeps = append(rootDeps, *dep)
		}
	}

	return manager.Module{
		Name:         moduleName,
		Version:      moduleVersion,
		Dependencies: rootDeps,
		LocalPath:    opts.targetPath,
	}, nil
}

func (g graph) resolveDependencyName(name string) *Module {
	parsed, err := parseDependencyName(name)
	if err != nil {
		return nil
	}
	for _, dep := range g {
		if parsed.Name() == dep.Name() {
			return &dep
		}
	}
	return nil
}

func (g graph) resolve(m Module, withDev bool) (*manager.Dependency, error) {
	name := m.Name()
	resolvedVersion, err := versionarium.NewVersion(language, m.version)
	if err != nil {
		return nil, err
	}
	resolvedDeps := []manager.Dependency{}
	for _, depName := range m.dependencies {
		mod := g.resolveDependencyName(depName)
		if mod.IsDev() && !withDev || mod.HasClassifierSpecification() {
			continue
		}
		if mod != nil {
			dep, err := g.resolve(*mod, withDev)
			if err != nil {
				simplelog.Error("failed to resolve dependency", "name", mod.raw, "err", err)
			} else {
				resolvedDeps = append(resolvedDeps, *dep)
			}
		} else {
			simplelog.Error("Dependency not found", "name", depName)
		}
	}
	return &manager.Dependency{
		Name:                 name,
		RawVersions:          m.version,
		ResolvedVersion:      resolvedVersion,
		Language:             language,
		IsDev:                m.IsDev(),
		ResolvedDependencies: resolvedDeps,
	}, nil

}

// https://github.com/bazelbuild/rules_jvm_external/blob/master/specs.bzl
func parseDependencyName(name string) (*coordinates, error) {
	params := strings.Split(name, ":")
	parsed := coordinates{
		raw: name,
	}

	switch len(params) {
	case 3:
		// groupID:artifactID:version
		parsed.groupID = params[0]
		parsed.artifactID = params[1]
		parsed.version = params[2]
	case 4:
		// groupId:artifactId:packaging:version
		parsed.groupID = params[0]
		parsed.artifactID = params[1]
		parsed.packaging = params[2]
		parsed.version = params[3]
	case 5:
		// groupId:artifactId:packaging:classifier:version
		parsed.groupID = params[0]
		parsed.artifactID = params[1]
		parsed.packaging = params[2]
		parsed.classifier = params[3]
		parsed.version = params[4]
	default:
		return nil, fmt.Errorf("bad fields number: %s", name)
	}
	return &parsed, nil
}

func parseMavenInstallJSON(target string) (graph, error) {
	dummy := []Module{}
	if !fsutils.IsFileExists(target) {
		return dummy, fmt.Errorf("file not found: %s", target)
	}

	file, err := ioutil.ReadFile(target)
	if err != nil {
		return dummy, fmt.Errorf("cannot open file: %w", err)
	}

	parsedFile := mavenInstallJSON{}
	err = json.Unmarshal([]byte(file), &parsedFile)
	if err != nil {
		return dummy, fmt.Errorf("cannot unmarshal file: %w", err)
	}

	return parsedFile.Tree.Dependencies, nil
}
