package generic

import (
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"os"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/versionarium"
	"a.yandex-team.ru/security/yadi/yadi/pkg/analyze"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager"
)

const name = "generic"

type (
	ManagerOpts struct {
		// Target file to parse
		TargetPath string
		WithDev    bool
	}
	Manager struct {
		target  string
		lang    string
		withDev bool //TODO(melkikh): think about dev deps
		trees   []analyze.ModuleTree
	}
)

var _ manager.PackageManager = (*Manager)(nil)

func NewManager(opts ManagerOpts) (*Manager, error) {
	if opts.TargetPath == "" {
		return nil, errors.New("empty target file path")
	}

	if _, err := os.Stat(opts.TargetPath); os.IsNotExist(err) {
		return nil, fmt.Errorf("target %s not found", opts.TargetPath)
	}

	projectInfo, err := readProjectInfo(opts.TargetPath)
	if err != nil {
		return nil, fmt.Errorf("failed to parse project info: %w", err)
	}
	return &Manager{
		target:  opts.TargetPath,
		lang:    projectInfo.Language,
		withDev: opts.WithDev,
		trees:   projectInfo.Trees,
	}, nil
}

func (m *Manager) Name() string {
	return name
}

func (m *Manager) Language() string {
	return m.lang
}

func (m *Manager) TargetPath() string {
	return m.target
}

func (m *Manager) CanLocal() bool {
	return true
}

func (m *Manager) CanRemote() bool {
	return false
}

func (m *Manager) CanSuggest() bool {
	return false
}

func (m *Manager) Cacheable() bool {
	return false
}

func (m *Manager) RootModules() ([]manager.Module, error) {
	result := make([]manager.Module, 0, len(m.trees))
	for _, subprojTree := range m.trees {
		module, err := m.resolveModules(subprojTree)
		if err != nil {
			simplelog.Error("failed to resolve module", "project", subprojTree.PackageName, "err", err)
			break
		}
		result = append(result, module)
	}
	return result, nil
}

func (m *Manager) ResolveLocalDependency(dep manager.Dependency, _ manager.Module) (manager.Module, error) {
	return manager.Module{
		Name:         dep.Name,
		Version:      dep.ResolvedVersion,
		Dependencies: dep.ResolvedDependencies,
		LocalPath:    dep.LocalPath,
	}, nil
}

func (m *Manager) ResolveRemoteDependency(_ manager.Dependency, _ manager.Module) (manager.Module, error) {
	panic("not implemented: must not be called")
}

func (m *Manager) SuggestModuleUpdate(_ manager.Module, _ versionarium.VersionRange, _ []manager.Module) []string {
	return nil
}

func readProjectInfo(path string) (analyze.ProjectInfo, error) {
	var project analyze.ProjectInfo
	data, err := ioutil.ReadFile(path)
	if err != nil {
		return project, err
	}

	err = json.Unmarshal(data, &project)
	if err != nil {
		return project, err
	}

	return project, nil
}

func (m *Manager) resolveModules(tree analyze.ModuleTree) (manager.Module, error) {
	version, err := versionarium.NewVersion(m.lang, tree.Version)
	if err != nil {
		return manager.ZeroModule, fmt.Errorf("failed to parse version: %w", err)
	}

	return manager.Module{
		Name:         tree.PackageName,
		LocalPath:    tree.Path,
		Version:      version,
		License:      tree.License,
		Dependencies: m.resolveDeps(tree.Dependencies),
	}, nil
}

func (m *Manager) resolveDeps(deps []analyze.ModuleTree) []manager.Dependency {
	result := make([]manager.Dependency, 0)
	for _, dep := range deps {
		version, err := versionarium.NewVersion(m.lang, dep.Version)
		if err != nil {
			simplelog.Error("failed to parse version", "dependency", dep.Version, "err", err)
			continue
		}
		result = append(result, manager.Dependency{
			Name:                 dep.PackageName,
			RawVersions:          dep.Version,
			IsDev:                false,
			LocalPath:            dep.Path,
			Language:             m.lang,
			ResolvedVersion:      version,
			ResolvedDependencies: m.resolveDeps(dep.Dependencies),
		})
	}
	return result
}
