package gomod

import (
	"bytes"
	"context"
	"encoding/json"
	"io"
	"os"
	"os/exec"
	"path/filepath"
	"strings"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

type goPackage struct {
	Name        string
	Version     string
	Imports     []string
	TestImports []string
}

type nativePackage struct {
	ImportPath string
	Module     *nativeModule

	Imports      []string
	TestImports  []string
	XTestImports []string
}

type nativeModule struct {
	Path    string
	Version string
	Main    bool
}

func parseGoList(reader io.Reader) ([]nativePackage, error) {
	var pkgs []nativePackage
	decoder := json.NewDecoder(reader)

	for {
		var pkg nativePackage
		err := decoder.Decode(&pkg)
		if err != nil && err != io.EOF {
			return nil, xerrors.Errorf("can't parse go output: %w", err)
		} else if err == io.EOF {
			break
		}

		pkgs = append(pkgs, pkg)
	}

	return pkgs, nil
}

func runGoList(goModPath string) ([]nativePackage, error) {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	simplelog.Info("lookup dependencies", "command", "go list --mod=readonly --json --deps ./...")
	defer simplelog.Info("lookup done")

	cmd := exec.CommandContext(ctx, "go", "list", "--mod=readonly", "--json", "--deps", "./...")
	cmd.Dir = filepath.Dir(goModPath)
	var stdout bytes.Buffer
	cmd.Stdout = &stdout
	cmd.Stderr = os.Stderr
	// TODO(buglloc): do we need other OSes?
	cmd.Env = append(os.Environ(), "GOOS=linux")

	if err := cmd.Run(); err != nil {
		return nil, xerrors.Errorf("failed to run go: %w", err)
	}

	return parseGoList(&stdout)
}

func compactGoPackages(nativePackages []nativePackage) map[string]goPackage {
	clearImports := func(src []string) []string {
		var out []string
		for _, imp := range src {
			if !strings.ContainsRune(imp, '.') {
				// not external pkg
				continue
			}

			out = append(out, imp)
		}
		return out
	}

	out := make(map[string]goPackage)
	for _, nativePkg := range nativePackages {
		if nativePkg.Module == nil || nativePkg.Module.Path == "" {
			// stdlib
			continue
		}

		out[nativePkg.ImportPath] = goPackage{
			Name:    nativePkg.Module.Path,
			Version: nativePkg.Module.Version,
			Imports: clearImports(nativePkg.Imports),
			TestImports: append(
				clearImports(nativePkg.TestImports),
				clearImports(nativePkg.XTestImports)...,
			),
		}
	}
	return out
}
