package gomod

import (
	"io/ioutil"
	"os"

	"golang.org/x/mod/modfile"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/versionarium"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager"
)

const (
	name     = "gomod"
	language = "golang"
)

var _ manager.PackageManager = (*Manager)(nil)
var nilVersion, _ = versionarium.NewVersion(language, "v0.0.0")

type (
	ManagerOpts struct {
		// Target go.mod
		TargetPath string

		// Name of root package (for NewManagerFromGoListFile)
		RootPkgName string

		// Parse dev dependency
		WithDev bool
	}

	Manager struct {
		pkgs    map[string]goPackage
		target  string
		rootPkg string
		withDev bool
	}
)

func NewManager(opts ManagerOpts) (*Manager, error) {
	goModData, err := ioutil.ReadFile(opts.TargetPath)
	if err != nil {
		return nil, xerrors.Errorf("failed to read go.mod: %w", err)
	}

	rootPkg := modfile.ModulePath(goModData)
	if rootPkg == "" {
		return nil, xerrors.Errorf("no module path in go.mod: %s", opts.TargetPath)
	}

	goPkgs, err := runGoList(opts.TargetPath)
	if err != nil {
		return nil, xerrors.Errorf("go list fail: %w", err)
	}

	return &Manager{
		pkgs:    compactGoPackages(goPkgs),
		target:  opts.TargetPath,
		rootPkg: rootPkg,
		withDev: opts.WithDev,
	}, nil
}

func NewManagerFromGoListFile(opts ManagerOpts) (*Manager, error) {
	targetFile, err := os.Open(opts.TargetPath)
	if err != nil {
		return nil, xerrors.Errorf("failed to open targetFile %s: %w", opts.TargetPath, err)
	}
	defer func() { _ = targetFile.Close() }()

	goPkgs, err := parseGoList(targetFile)
	if err != nil {
		return nil, xerrors.Errorf("can't read 'go list' output: %w", err)
	}

	return &Manager{
		pkgs:    compactGoPackages(goPkgs),
		target:  opts.TargetPath,
		rootPkg: opts.RootPkgName,
		withDev: opts.WithDev,
	}, nil
}

func (m *Manager) Name() string {
	return name
}

func (m *Manager) Language() string {
	return language
}

func (m *Manager) TargetPath() string {
	return m.target
}

func (m *Manager) Cacheable() bool {
	return true
}

func (m *Manager) CanLocal() bool {
	return true
}

func (m *Manager) CanRemote() bool {
	return false
}

func (m *Manager) CanSuggest() bool {
	return false
}

func (m *Manager) RootModules() ([]manager.Module, error) {
	imports := make(map[string]struct{})
	for i, pkg := range m.pkgs {
		if pkg.Name != m.rootPkg {
			continue
		}

		if len(pkg.Imports) == 0 && (!m.withDev || len(pkg.TestImports) == 0) {
			continue
		}

		imports[i] = struct{}{}
	}

	rootModule := manager.Module{
		Name:      m.rootPkg,
		LocalPath: m.target,
		Version:   nilVersion,
	}

	for i := range imports {
		rootModule.Dependencies = append(rootModule.Dependencies, manager.Dependency{
			Name: i,
		})
	}

	return []manager.Module{rootModule}, nil
}

func (m *Manager) ResolveLocalDependency(dep manager.Dependency, _ manager.Module) (manager.Module, error) {
	pkg, ok := m.pkgs[dep.Name]
	if !ok {
		return manager.ZeroModule, xerrors.Errorf("no import path '%s' found", dep.Name)
	}

	module := manager.Module{
		Name: dep.Name,
	}

	if pkg.Version == "" {
		module.Version = nilVersion
	} else {
		v, err := versionarium.NewVersion(language, pkg.Version)
		if err != nil {
			return module, xerrors.Errorf("failed to parse pkg version: %w", err)
		}
		module.Version = v
	}

	imports := make(map[string]struct{})
	for _, i := range pkg.Imports {
		imports[i] = struct{}{}
	}

	if m.withDev {
		for _, i := range pkg.TestImports {
			imports[i] = struct{}{}
		}
	}

	for i := range imports {
		iPkg, ok := m.pkgs[i]
		if !ok {
			simplelog.Debug("ignore self-pkg dep", "import_path", i)
			continue
		}

		if iPkg.Name == m.rootPkg {
			simplelog.Debug("ignore self-pkg dep", "import_path", i)
			continue
		}

		module.Dependencies = append(module.Dependencies, manager.Dependency{
			Name: i,
		})
	}

	return module, nil
}

func (m *Manager) ResolveRemoteDependency(_ manager.Dependency, _ manager.Module) (manager.Module, error) {
	panic("not implemented: must not be called")
}

func (m *Manager) SuggestModuleUpdate(_ manager.Module, _ versionarium.VersionRange, _ []manager.Module) []string {
	return nil
}
