package manager

import (
	"a.yandex-team.ru/security/yadi/libs/versionarium"
)

const (
	ResolveLocal ResolveMode = 1 << iota
	ResolveRemote
)

type (
	// Интерфейс пакетного менеджера, с которым работает анализатор.
	// Сам анализатор двухэтапный:
	//  - сначала резолвим дерево зависимостей
	//  - потом ищем в нем уязвимости
	//
	// Взаимодействие анализатора с пакетным менеджером при резолве зависимостей:
	//   1. анализатор запрашивает список top-level пакетиков (RootModules)
	//   2. бежит по зависимостям каждого из них
	//   3. если пакетный менеджер умеет резолвить пакетики локально (CanLocal) - анализатор просит порезолвить (ResolveLocalDependency)
	//   4. если локальный резолв пофейлился (или пакетный менеджер этого не умеет), узнает можно ли порезолвить удаленно (CanRemote). Если можно - пробует это сделать (ResolveRemoteDependency)
	//   5. если и тут все пофейлилось - анализатор считает что порезолвить не удалось и скипует эту зависимость
	//   6. переходит к п.2 но для зависимостей этого модуля
	//
	// Взаимодействие анализатора с пакетным менеджером при поиске уязвимостей:
	//   1. если анализатор нашел багу, то просит пакетный менеджер подсказать потребителю на что её обновлять (SuggestModuleUpdate)
	//
	// Для типичного, упрощенного пакетного менеджера нужно реализовать:
	//   - Name
	//   - Language
	//   - RootModules
	//   - CanLocal
	//   - ResolveLocalDependency
	//
	PackageManager interface {
		// Имя пакетного менеджера
		Name() string

		// Язык с которым он работает
		Language() string

		// Путь к используемой цели
		TargetPath() string

		// Можно ли кешировать поддеревья
		Cacheable() bool

		// Возвращает список top-level модулей
		RootModules() ([]Module, error)

		// Умеет ли пакетный менеджер _локально_ резолвить пакетики (у него можно вызывать ResolveLocalDependency)
		CanLocal() bool

		// Порезолвить пакетик локально
		ResolveLocalDependency(dep Dependency, module Module) (Module, error)

		// Умеет ли пакетный менеджер _удаленно_ резолвить пакетики (у него можно вызывать ResolveRemoteDependency)
		CanRemote() bool

		// Порезолвить пакетик удаленно
		ResolveRemoteDependency(dep Dependency, module Module) (Module, error)

		// Умеет ли пакетный менеджер выдавать подсказки, это напрямую влияет на возможность кеширования уязвимостей
		CanSuggest() bool

		// Подсказать на что обновлять пакетик
		SuggestModuleUpdate(module Module, vulnerableVersions versionarium.VersionRange, parents []Module) []string
	}

	ResolveMode int
)
