package maven

import (
	"bufio"
	"context"
	"fmt"
	"io"
	"io/ioutil"
	"os"
	"os/exec"
	"path"
	"strings"

	"github.com/awalterschulze/gographviz"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/versionarium"
	"a.yandex-team.ru/security/yadi/yadi/pkg/fsutils"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager"
)

type (
	graph       gographviz.Graph
	graphList   []graph
	mavenModule struct {
		// https://maven.apache.org/plugins/maven-dependency-plugin/examples/filtering-the-dependency-tree.html
		// [groupId]:[artifactId]:[type]:[version]
		raw, groupID, artifactID, moduleType, version, status string
	}

	resolveOpts struct {
		withDev bool
	}
)

func runMvnDepsTree(target string) (graphList, error) {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	if _, err := os.Stat(target); os.IsNotExist(err) {
		return nil, xerrors.Errorf("target does not exist: %s", target)
	}

	targetDir := path.Dir(target)
	dotFile, err := ioutil.TempFile("", "mvn-deps.*.dot")
	defer func() { _ = dotFile.Close() }()
	if err != nil {
		return nil, xerrors.Errorf("failed to open file: %w", err)
	}

	args := []string{"dependency:tree", "-DoutputType=dot", "-DappendOutput=true", fmt.Sprintf("-DoutputFile=%s", dotFile.Name())}

	simplelog.Info(fmt.Sprintf("running command 'mvn %s'", strings.Join(args, " ")))
	cmd := exec.CommandContext(ctx, "mvn", args...)
	cmd.Dir = targetDir
	cmd.Stderr = os.Stderr
	cmd.Stdout = os.Stderr

	if err := cmd.Run(); err != nil {
		return nil, xerrors.Errorf("failed to run mvn: %w", err)
	}
	defer func() { _ = os.Remove(dotFile.Name()) }()
	return readDotFile(dotFile.Name())
}

func readDotFile(path string) (graphList, error) {
	if !fsutils.IsFileExists(path) {
		return nil, xerrors.Errorf("file not found: %s", path)
	}

	file, err := os.Open(path)
	defer func() { _ = file.Close() }()
	if err != nil {
		return nil, xerrors.Errorf("cannot open file: %w", err)
	}

	rawGraphList := make([][]byte, 0)
	reader := bufio.NewReader(file)

	err = splitDigraphs(reader, &rawGraphList)
	if err != nil {
		return nil, xerrors.Errorf("failed to split DOT-graph: %w", err)
	}

	var result graphList
	for _, rawGraph := range rawGraphList {
		digraph, err := gographviz.Read(rawGraph)
		if err != nil {
			return nil, xerrors.Errorf("failed to parse DOT-graph: %w", err)
		}
		result = append(result, graph(*digraph))
	}

	return result, nil
}

func splitDigraphs(r *bufio.Reader, out *[][]byte) error {
	for {
		chunk, err := r.ReadBytes('}')
		if err != nil {
			if err != io.EOF {
				return err
			}
			break
		}
		*out = append(*out, chunk)
	}
	return nil
}

func parseModule(raw string) (mavenModule, error) {
	unquoted := raw[1 : len(raw)-1] // remove quotes
	params := strings.Split(unquoted, ":")
	if len(params) < 4 {
		return mavenModule{}, xerrors.Errorf("bad fields number: %s", unquoted)
	}

	var status string
	if len(params) == 5 {
		status = params[4]
	}

	return mavenModule{
		raw:        raw,
		groupID:    params[0],
		artifactID: params[1],
		moduleType: params[2],
		version:    params[3],
		status:     status,
	}, nil
}

func (g graph) resolveTree(opts resolveOpts) (manager.Module, error) {
	root, err := parseModule(g.Name)
	if err != nil {
		return manager.ZeroModule, xerrors.Errorf("failed to parse module: %w", err)
	}

	version, err := versionarium.NewVersion(language, root.version)
	if err != nil {
		return manager.ZeroModule, xerrors.Errorf("failed to parse version for module %s: %w", root.Name(), err)
	}

	return manager.Module{
		Name:         root.Name(),
		Version:      version,
		Dependencies: g.resolveDependencies(root, opts.withDev),
		LocalPath:    root.raw,
	}, nil
}

func (g graph) resolveDependencies(parent mavenModule, withDev bool) []manager.Dependency {
	deps, ok := g.Edges.SrcToDsts[parent.raw]
	if !ok {
		simplelog.Debug("there are no dependencies for module", "module", parent.Name())
		return nil
	}

	resolvedDeps := make([]manager.Dependency, 0, len(deps))
	for dep := range deps {
		depModule, err := parseModule(dep)
		if err != nil {
			simplelog.Error("failed to resolve dependency", "dep", dep, "err", err)
			break
		}

		version, err := versionarium.NewVersion(language, depModule.version)
		if err != nil {
			simplelog.Error("failed to parse version", "dependency", depModule.Name(), "err", err)
			continue
		}

		if depModule.IsDev() && !withDev {
			continue
		}

		resolvedDeps = append(resolvedDeps, manager.Dependency{
			Name:                 depModule.Name(),
			RawVersions:          depModule.version,
			ResolvedVersion:      version,
			Language:             language,
			IsDev:                depModule.IsDev(),
			ResolvedDependencies: g.resolveDependencies(depModule, withDev),
		})
	}
	return resolvedDeps
}

func (m mavenModule) Name() string {
	return fmt.Sprintf("%s:%s", m.groupID, m.artifactID)
}

func (m mavenModule) IsDev() bool {
	switch m.status {
	case "test":
		return true
	case "compile", "provided", "runtime":
		fallthrough
	default:
		return false
	}
}
