package maven

import (
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/versionarium"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager"
)

const (
	name     = "maven"
	language = "java"
)

var _ manager.PackageManager = (*Manager)(nil)

type (
	ManagerOpts struct {
		// Target file to parse
		TargetPath string
		WithDev    bool
	}

	Manager struct {
		target  string
		withDev bool
		graphs  graphList
	}
)

func NewManager(opts ManagerOpts) (*Manager, error) {
	graphs, err := runMvnDepsTree(opts.TargetPath)
	if err != nil {
		return nil, err
	}
	return &Manager{
		graphs:  graphs,
		target:  opts.TargetPath,
		withDev: opts.WithDev,
	}, nil
}

func NewManagerFromDotFile(opts ManagerOpts) (*Manager, error) {
	graphs, err := readDotFile(opts.TargetPath)
	if err != nil {
		return nil, err
	}
	return &Manager{
		graphs:  graphs,
		target:  opts.TargetPath,
		withDev: opts.WithDev,
	}, nil
}

func (m *Manager) Name() string {
	return name
}

func (m *Manager) Language() string {
	return language
}

func (m *Manager) TargetPath() string {
	return m.target
}

func (m *Manager) CanLocal() bool {
	return true
}

func (m *Manager) CanRemote() bool {
	return false
}

func (m *Manager) CanSuggest() bool {
	return false
}

func (m *Manager) Cacheable() bool {
	return false
}

func (m *Manager) RootModules() ([]manager.Module, error) {
	result := make([]manager.Module, 0, len(m.graphs))
	for _, digraph := range m.graphs {
		module, err := digraph.resolveTree(resolveOpts{withDev: m.withDev})
		if err != nil {
			simplelog.Error("failed to resolve module", "digraph", digraph.Name, "err", err)
			break
		}
		result = append(result, module)
	}

	return result, nil
}

func (m *Manager) ResolveLocalDependency(dep manager.Dependency, _ manager.Module) (manager.Module, error) {
	return manager.Module{
		Name:         dep.Name,
		Version:      dep.ResolvedVersion,
		Dependencies: dep.ResolvedDependencies,
		LocalPath:    m.target,
	}, nil
}

func (m *Manager) ResolveRemoteDependency(_ manager.Dependency, _ manager.Module) (manager.Module, error) {
	panic("not implemented: must not be called")
}

func (m *Manager) SuggestModuleUpdate(_ manager.Module, _ versionarium.VersionRange, _ []manager.Module) []string {
	return nil
}
