package npm

import (
	"errors"
	"fmt"
	"io/ioutil"
	"net/url"
	"path/filepath"
	"strings"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/versionarium"
	"a.yandex-team.ru/security/yadi/yadi/internal/config"
	"a.yandex-team.ru/security/yadi/yadi/internal/httputils"
	"a.yandex-team.ru/security/yadi/yadi/pkg/fsutils"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager"
)

const lang = "nodejs"

var _ manager.PackageManager = (*Manager)(nil)

type (
	ManagerOpts struct {
		// Target file to parse
		TargetPath string

		// Parse dev dependency
		WithDev bool

		// Dependency resolve mode
		// Default: local & remote
		ResolveMode manager.ResolveMode
	}

	Manager struct {
		target      string
		withDev     bool
		resolveMode manager.ResolveMode
	}
)

func NewManager(opts ManagerOpts) (*Manager, error) {
	if opts.ResolveMode == 0 {
		opts.ResolveMode = manager.ResolveLocal | manager.ResolveRemote
	}

	if opts.TargetPath == "" {
		opts.ResolveMode &^= manager.ResolveLocal
	} else {
		if !fsutils.IsFileExists(opts.TargetPath) {
			return nil, errors.New("target file not exists")
		}

		if opts.ResolveMode&manager.ResolveLocal != 0 {
			modulesPath := filepath.Join(filepath.Dir(opts.TargetPath), "node_modules")
			if !fsutils.IsDirExists(modulesPath) {
				simplelog.Debug("disable local resolver, 'node_modules' not exists")
				opts.ResolveMode &^= manager.ResolveLocal
			}
		}
	}

	if opts.ResolveMode&manager.ResolveLocal == 0 && opts.ResolveMode&manager.ResolveRemote == 0 {
		return nil, errors.New("NPM package manager works only as a Local or Remote resolver")
	}

	return &Manager{
		target:      opts.TargetPath,
		withDev:     opts.WithDev,
		resolveMode: opts.ResolveMode,
	}, nil
}

func (m *Manager) Name() string {
	return "npm"
}

func (m *Manager) Language() string {
	return lang
}

func (m *Manager) TargetPath() string {
	return m.target
}

func (m *Manager) Cacheable() bool {
	return false
}

func (m *Manager) CanLocal() bool {
	return m.resolveMode&manager.ResolveLocal != 0
}

func (m *Manager) CanRemote() bool {
	return m.resolveMode&manager.ResolveRemote != 0
}

func (m *Manager) CanSuggest() bool {
	return true
}

func (m *Manager) RootModules() ([]manager.Module, error) {
	return m.ResolveRootWithSubmodules(m.target)
}

func (m *Manager) ResolveLocalDependency(dep manager.Dependency, parent manager.Module) (manager.Module, error) {

	var path string
	if dep.IsStrictlyLocal() {
		path = dep.LocalPath
	} else {
		basePath := filepath.Dir(parent.LocalPath)
		for {
			tmpPath := filepath.Join(basePath, "node_modules", dep.Name, "package.json")
			if fsutils.IsFileExists(tmpPath) {
				path = tmpPath
				break
			}
			i := strings.LastIndex(basePath, "node_modules")
			if i <= 0 {
				break
			}
			basePath = basePath[:i-1]
		}

		if path == "" {
			return manager.ZeroModule, fmt.Errorf("package %s not found", dep.Name)
		}
	}

	result, err := m.resolveModule(path)
	if err != nil {
		return result, err
	}

	if !dep.IsStrictlyLocal() && !dep.CheckVersion(result.Version) {
		simplelog.Error(fmt.Sprintf("Dependency %s not satisfies by package version: %s != %s",
			path, dep.RawVersions, result.Version))
	}

	return result, nil
}

func (m *Manager) ResolveRemoteDependency(dep manager.Dependency, parent manager.Module) (manager.Module, error) {
	if dep.IsStrictlyLocal() {
		// If we have a strictly local dependency (e.g. file:../module) we must resolve it locally!
		return m.ResolveLocalDependency(dep, parent)
	}

	packageVersions, err := FetchPackageVersions(dep.Name)
	if err != nil {
		return manager.ZeroModule, err
	}

	var (
		maxVer versionarium.Version
		maxID  = -1
	)
	for i, meta := range packageVersions {
		if !dep.CheckVersion(meta.Version) {
			continue
		}

		if maxVer != nil && maxVer.GreaterThan(meta.Version) {
			continue
		}

		maxVer = meta.Version
		maxID = i
	}

	if maxID == -1 {
		return manager.ZeroModule, fmt.Errorf("failed to find dependency %s", dep.FullName())
	}
	return packageVersions[maxID].NewModule(false), nil
}

func (m *Manager) SuggestModuleUpdate(module manager.Module, vulnerableVersions versionarium.VersionRange, parents []manager.Module) []string {
	currentVersions, err := FetchPackageVersions(module.Name)
	if err != nil {
		simplelog.Warn("failed to fetch module versions to suggest updates", "module", module.Name, "err", err)
		return nil
	}

	// we always have "root" pkg, so we must to have more than 2 parents for transitive dep
	isDirect := len(parents) < 2
	var fixedVersions []versionarium.Version
	for _, candidate := range currentVersions {
		if candidate.Version.ReleaseInfo() != "" {
			// Skip prerelease versions
			continue
		}

		if candidate.Version.LessThan(module.Version) {
			// Not needed
			continue
		}

		if vulnerableVersions.Check(candidate.Version) {
			// Vulnerable version
			continue
		}

		fixedVersions = append(fixedVersions, candidate.Version)
	}

	if len(fixedVersions) == 0 {
		return nil
	}

	currentSuggest := []string{fmt.Sprintf("%s@%s", module.Name, fixedVersions[0].String())}
	if isDirect {
		return currentSuggest
	}

	// Transitive dependency
	parent := parents[len(parents)-1]
	parentVersions, err := FetchPackageVersions(parent.Name)
	if err != nil {
		simplelog.Warn("failed to fetch module versions to suggest updates", "module", parent.Name, "err", err)
		return nil
	}

	var parentVulnRangeBuilder strings.Builder
	for _, candidate := range parentVersions {
		if candidate.Version.ReleaseInfo() != "" {
			// Skip prerelease versions
			continue
		}

		if candidate.Version.LessThan(parent.Version) {
			// Not needed
			continue
		}

		for name, ver := range candidate.Dependencies {
			if name != module.Name {
				// Not interesting dep
				continue
			}

			versions, err := versionarium.NewRange(lang, ver)
			if err != nil {
				simplelog.Error("failed to parse parent dependency version",
					"module", name, "version", ver, "err", err)
				continue
			}

			canFix := false
			for _, ver := range fixedVersions {
				if versions.Check(ver) {
					canFix = true
					break
				}
			}

			if !canFix {
				continue
			}

			if parentVulnRangeBuilder.Len() > 0 {
				parentVulnRangeBuilder.WriteByte(' ')
			}
			parentVulnRangeBuilder.WriteString("!=")
			parentVulnRangeBuilder.WriteString(candidate.RawVersion)
			break
		}
	}

	parentVulnVersion := parentVulnRangeBuilder.String()
	if parentVulnVersion == "" {
		return nil
	}

	parentVulnRange, err := versionarium.NewRange(lang, parentVulnVersion)
	if err != nil {
		simplelog.Error("failed to parse parent version range",
			"module", module.String(), "parent", parent.String(), "versions", parentVulnRange, "err", err)
		return nil
	}

	parentVersion := m.SuggestModuleUpdate(parent, parentVulnRange, parents[:len(parents)-1])
	if len(parentVersion) != 0 {
		return append(parentVersion, currentSuggest...)
	}

	return nil
}

func (m *Manager) resolveModule(path string) (manager.Module, error) {
	pj, err := ParsePackageJSONFile(path, true)
	if err != nil {
		return manager.ZeroModule, err
	}

	return pj.NewModule(m.withDev), nil
}

func (m *Manager) ResolveRootWithSubmodules(path string) ([]manager.Module, error) {
	pj, err := ParsePackageJSONFile(path, true)
	if err != nil {
		return nil, err
	}

	result := []manager.Module{
		pj.NewModule(m.withDev),
	}

	if len(pj.Workspaces) > 0 {
		workspaces, err := pj.ResolveWorkspaces()
		if err != nil {
			return nil, err
		}

		for _, w := range workspaces {
			resolved, err := m.resolveModule(w)
			if err != nil {
				simplelog.Error("failed to resolve workspace", "path", w, "err", err.Error())
				continue
			}
			result = append(result, resolved)
		}
	}

	return result, nil
}

func fetchNpm(data ...string) ([]byte, error) {
	simplelog.Debug("fetch package: " + strings.Join(data, ":"))

	reURL, _ := url.Parse(config.NpmRepositoryURI)
	reURL.Path = strings.Join(data, "/")
	resp, err := httputils.DoGet(reURL.String())
	if err != nil {
		return nil, err
	}
	defer httputils.GracefulClose(resp.Body)

	if resp.StatusCode != 200 {
		return nil, errors.New("package not found")
	}

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, err
	}

	return body, nil
}
