package npm

//go:generate easyjson

import (
	"errors"
	"fmt"
	"io/ioutil"
	"path"
	"path/filepath"
	"strings"

	"github.com/mailru/easyjson"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/versionarium"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager"
)

type (
	// License is so ugly due two multiple npm License formats :(

	//easyjson:json
	PackageJSON struct {
		Name            string               `json:"name"`
		Description     string               `json:"description"`
		RawVersion      string               `json:"version"`
		Workspaces      []string             `json:"workspaces"`
		License         interface{}          `json:"license"`
		Version         versionarium.Version `json:"-"`
		LocalPath       string               `json:"-"`
		Dependencies    map[string]string    `json:"dependencies"`
		DevDependencies map[string]string    `json:"devDependencies"`
		OptDependencies map[string]string    `json:"optionalDependencies"`
	}
)

func ParsePackageJSONFile(path string, localPath bool) (pj *PackageJSON, err error) {
	data, err := ioutil.ReadFile(path)
	if err != nil {
		return
	}

	pj, err = ParsePackageJSON(data)
	if err == nil && localPath {
		pj.LocalPath = path
	}
	return
}

func ParsePackageJSON(packageInfo []byte) (*PackageJSON, error) {
	var pj PackageJSON
	err := easyjson.Unmarshal(packageInfo, &pj)
	if err != nil {
		return nil, err
	}

	if pj.Name == "" {
		simplelog.Warn("no package name specified, using 'unspecified'")
		pj.Name = "unspecified"
	}

	if pj.RawVersion == "" {
		simplelog.Warn(fmt.Sprintf("Package '%s' have no version specified", pj.Name))
	}

	return &pj, err
}

func (p *PackageJSON) NewModule(withDev bool) manager.Module {
	var moduleLicense string
	switch l := p.License.(type) {
	case string:
		moduleLicense = l
	case map[string]string:
		moduleLicense = l["type"]
	}

	module := manager.Module{
		Name:      p.Name,
		License:   moduleLicense,
		LocalPath: p.LocalPath,
	}

	if p.Version != nil {
		module.Version = p.Version
	} else if p.RawVersion != "" {
		var err error
		module.Version, err = versionarium.NewVersion(lang, p.RawVersion)
		if err != nil {
			simplelog.Error("failed to parse version: "+err.Error(),
				"module", p.Name, "version", p.RawVersion)
		}
	} else {
		simplelog.Warn("no package version specified", "module", p.Name)
	}

	module.Dependencies = p.appendDependencies(module.Dependencies, p.Dependencies, false)
	module.Dependencies = p.appendDependencies(module.Dependencies, p.OptDependencies, false)
	if withDev {
		module.Dependencies = p.appendDependencies(module.Dependencies, p.DevDependencies, true)
	}

	return module
}

func (p *PackageJSON) String() string {
	return fmt.Sprintf("%s@%s", p.Name, p.Version.String())
}

func (p *PackageJSON) appendDependencies(target []manager.Dependency, source map[string]string, isDev bool) []manager.Dependency {
	for name, ver := range source {
		if strings.Index(ver, "http") == 0 || strings.Index(ver, "git") == 0 {
			simplelog.Warn("skip dependency", "module", name, "version", ver)
			continue
		}

		//TODO(buglloc): refactor
		var localPath string
		rawVersion := ver
		if strings.HasPrefix(ver, "file:") {
			subPath, _, err := p.resolveLocalDep(ver[5:])
			if err != nil {
				simplelog.Warn("skip dependency: "+err.Error(), "module", name, "version", ver)
				continue
			}
			localPath = subPath
		}

		target = append(target, manager.Dependency{
			Name:        name,
			IsDev:       isDev,
			RawVersions: rawVersion,
			LocalPath:   localPath,
			Language:    lang,
		})
	}
	return target
}

func (p *PackageJSON) resolveLocalDep(dep string) (localPath string, version string, err error) {
	if p.LocalPath == "" {
		err = errors.New("failed to get local path")
		return
	}

	localPath = path.Join(path.Dir(p.LocalPath), dep, "package.json")
	pj, err := ParsePackageJSONFile(localPath, true)
	if err != nil {
		return
	}
	version = "=" + pj.RawVersion
	return
}

func (p *PackageJSON) ResolveWorkspaces() (paths []string, err error) {
	if len(p.Workspaces) == 0 {
		return
	}

	includes := make([]string, 0)
	excludes := make([]string, 0)
	basePath := p.LocalPath
	if basePath != "" {
		basePath, err = filepath.Abs(basePath)
		if err != nil {
			return
		}
		basePath = path.Dir(basePath)
	}

	for _, w := range p.Workspaces {
		if strings.HasPrefix(w, "!") {
			excludes = append(excludes, path.Join(basePath, w[1:]))
		} else {
			includes = append(includes, path.Join(basePath, w, "package.json"))
		}
	}

	collectedPaths := make(map[string]bool)
	for _, include := range includes {
		files, err := filepath.Glob(include)
		if err != nil {
			simplelog.Error("failed to search workspace", "include_pattern", include, "err", err)
			continue
		}

		for _, file := range files {
			skip := false
			for _, ex := range excludes {
				if matched, _ := filepath.Match(ex, file); matched {
					skip = true
					break
				}
			}

			if !skip {
				collectedPaths[file] = true
			}
		}
	}

	for p := range collectedPaths {
		paths = append(paths, p)
	}

	return
}
