package npmaudit

import (
	"fmt"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/yadi/libs/versionarium"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/npm"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/pkglock"
)

const (
	name = "npmaudit"
	lang = "nodejs"
)

var _ manager.PackageManager = (*Manager)(nil)

type (
	ManagerOpts struct {
		// Target PkgJSON
		PkgJSON *npm.PackageJSON

		// Target PkgLock
		PkgLockJSON *pkglock.LockJSON

		// Parse dev dependency
		WithDev bool
	}

	Manager struct {
		withDev bool
		pkgJSON *npm.PackageJSON
		pkgLock *pkglock.PackageLock
	}
)

func NewManager(opts ManagerOpts) (*Manager, error) {
	pkgLock, err := opts.PkgLockJSON.PackageLock(opts.WithDev)
	if err != nil {
		return nil, xerrors.Errorf("failed to parse pkg lock: %w", err)
	}

	return &Manager{
		withDev: opts.WithDev,
		pkgJSON: opts.PkgJSON,
		pkgLock: pkgLock,
	}, nil
}

func (m *Manager) Name() string {
	return name
}

func (m *Manager) Language() string {
	return lang
}

func (m *Manager) TargetPath() string {
	return "n/a"
}

func (m *Manager) Cacheable() bool {
	return false
}

func (m *Manager) CanLocal() bool {
	return true
}

func (m *Manager) CanRemote() bool {
	return false
}

func (m *Manager) CanSuggest() bool {
	return false
}

func (m *Manager) RootModules() ([]manager.Module, error) {
	if !m.withDev {
		m.pkgJSON.DevDependencies = nil

		for pkgName := range m.pkgJSON.Dependencies {
			if _, ok := m.pkgLock.Root.Dependencies[pkgName]; !ok {
				// remove dev deps
				delete(m.pkgJSON.Dependencies, pkgName)
			}
		}
	}

	return []manager.Module{
		m.pkgJSON.NewModule(m.withDev),
	}, nil
}

func (m *Manager) ResolveLocalDependency(dep manager.Dependency, _ manager.Module) (manager.Module, error) {
	if dep.ResolvedVersion == nil {
		// probably this is root module
		if module, ok := m.pkgLock.Root.Dependencies[dep.Name]; ok {
			return module.NewModule(), nil
		}
		return manager.ZeroModule, fmt.Errorf("failed to resolve local module '%s': empty resolved version", dep.Name)
	}

	// Otherwise this is already resolved dependency
	return manager.Module{
		Name:         dep.Name,
		Version:      dep.ResolvedVersion,
		Dependencies: dep.ResolvedDependencies,
	}, nil
}

// npm audit must return same module
// TODO(buglloc): strange logic
func (m *Manager) ResolveRemoteDependency(dep manager.Dependency, _ manager.Module) (manager.Module, error) {
	version, err := versionarium.NewVersion(lang, dep.RawVersions)
	if err != nil {
		return manager.ZeroModule, xerrors.Errorf("Failed to parse version '%s': %w", dep.RawVersions, err)
	}

	return manager.Module{
		Name:    dep.Name,
		Version: version,
	}, nil
}

func (m *Manager) SuggestModuleUpdate(_ manager.Module, _ versionarium.VersionRange, _ []manager.Module) []string {
	return nil
}
