package npmshrinkwrap

import (
	"errors"
	"io/ioutil"

	"a.yandex-team.ru/security/yadi/libs/versionarium"
	"a.yandex-team.ru/security/yadi/yadi/pkg/fsutils"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/npm"
)

const lang = "nodejs"

var _ manager.PackageManager = (*Manager)(nil)

type (
	ManagerOpts struct {
		// Target file to parse
		TargetPath string

		// Parse dev dependency
		WithDev bool
	}

	Manager struct {
		target     string
		npm        *npm.Manager
		withDev    bool
		shrinkJSON *ShrinkJSON
	}
)

func NewManager(opts ManagerOpts) (*Manager, error) {
	if opts.TargetPath != "" && !fsutils.IsFileExists(opts.TargetPath) {
		return nil, errors.New("target file not exists")
	}

	shrinkJSON, err := readShrinkJSON(opts.TargetPath)
	if err != nil {
		return nil, errors.New("failed to parse shrinkJSON")
	}

	npmManager, _ := npm.NewManager(npm.ManagerOpts{
		ResolveMode: manager.ResolveRemote,
		WithDev:     opts.WithDev,
	})

	return &Manager{
		target:     opts.TargetPath,
		npm:        npmManager,
		withDev:    opts.WithDev,
		shrinkJSON: shrinkJSON,
	}, nil
}

func (m *Manager) Name() string {
	return "npmshrinkwrap"
}

func (m *Manager) Language() string {
	return lang
}

func (m *Manager) TargetPath() string {
	return m.target
}

func (m *Manager) Cacheable() bool {
	return false
}

func (m *Manager) CanLocal() bool {
	return true
}

func (m *Manager) CanRemote() bool {
	return false
}

func (m *Manager) CanSuggest() bool {
	return false
}

func (m *Manager) RootModules() ([]manager.Module, error) {
	return []manager.Module{m.shrinkJSON.RootModule()}, nil
}

func (m *Manager) ResolveLocalDependency(dep manager.Dependency, _ manager.Module) (manager.Module, error) {
	return manager.Module{
		Name:    dep.Name,
		Version: dep.ResolvedVersion,
	}, nil
}

func (m *Manager) ResolveRemoteDependency(_ manager.Dependency, _ manager.Module) (manager.Module, error) {
	panic("not implemented: must not be called")
}

func (m *Manager) SuggestModuleUpdate(_ manager.Module, _ versionarium.VersionRange, _ []manager.Module) []string {
	// Unavailable due to lack of dependency tree
	return nil
}

func readShrinkJSON(path string) (*ShrinkJSON, error) {
	data, err := ioutil.ReadFile(path)
	if err != nil {
		return nil, err
	}

	result, err := ParseShrinkwrap(data)
	if err != nil {
		return nil, err
	}
	return result, nil
}
