package npmshrinkwrap

import (
	"encoding/json"
	"fmt"
	"strings"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/versionarium"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager"
)

type (
	/*
	   {
	     "name": "test",
	     "version": "0.0.0",
	     "dependencies": {
	       "send": {
	         "version": "0.3.0",
	         "from": "send@0.3.0",
	         "resolved": "http://storage.mds.yandex.net/get-npm/45674/send-0.3.0.tgz",
	         "dependencies": {
	           "debug": {
	             "version": "0.8.0",
	             "from": "debug@>=0.6 <1.0",
	             "resolved": "http://storage.mds.yandex.net/get-npm/38095/debug-0.8.0.tgz"
	           }
	         }
	       }
	     }
	   }
	*/

	ShrinkJSON struct {
		Name         string
		Version      string
		Dependencies map[string]ShrinkDep
	}

	ShrinkDep struct {
		Version      string
		From         string
		Dependencies map[string]ShrinkDep
	}
)

func (p *ShrinkJSON) RootModule() manager.Module {
	module := manager.Module{
		Name:         p.Name,
		Dependencies: flatShrinkDeps(p.Dependencies),
	}

	if p.Version != "" {
		version, err := versionarium.NewVersion(lang, p.Version)
		if err != nil {
			simplelog.Error("failed to parse version: "+err.Error(),
				"module", p.Name, "version", p.Version)
		}
		module.Version = version
	} else {
		simplelog.Warn(fmt.Sprintf("Package '%s' have no version specified", p.Name))
	}

	return module
}

func (p *ShrinkJSON) String() string {
	return fmt.Sprintf("%s@%s", p.Name, p.Version)
}

func ParseShrinkwrap(data []byte) (*ShrinkJSON, error) {
	var result ShrinkJSON
	err := json.Unmarshal(data, &result)
	if err != nil {
		return nil, err
	}

	if result.Name == "" {
		simplelog.Warn("no package name specified, using 'unspecified'")
		result.Name = "unspecified"
	}

	return &result, nil
}

func splitFrom(from string) (string, string) {
	if i := strings.LastIndex(from, "@"); i != -1 {
		return from[:i], from[i+1:]
	}
	return from, ""
}

func flatShrinkDeps(deps map[string]ShrinkDep) []manager.Dependency {
	result := make([]manager.Dependency, 0)
	for name, dep := range deps {
		var ver string
		if dep.From != "" {
			_, ver = splitFrom(dep.From)
		} else {
			ver = dep.Version
		}

		resolvedVersion, err := versionarium.NewVersion(lang, dep.Version)
		if err != nil {
			simplelog.Error("failed to parse version: "+err.Error(),
				"module", name, "version", ver)
			continue
		}

		result = append(result, manager.Dependency{
			Name:            name,
			RawVersions:     ver,
			ResolvedVersion: resolvedVersion,
			Language:        lang,
		})
		if len(dep.Dependencies) > 0 {
			result = append(result, flatShrinkDeps(dep.Dependencies)...)
		}
	}

	return result
}
