package local

import (
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"os"
	"os/exec"
	"path/filepath"
	"regexp"
	"strings"
	"sync"

	"a.yandex-team.ru/security/libs/go/pypi/pypipkg"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/pypi/pkgparser"
	"a.yandex-team.ru/security/yadi/libs/versionarium"
	"a.yandex-team.ru/security/yadi/yadi/pkg/fsutils"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/pip/requirements"
)

const (
	lang = "python"

	versionCv = `v?(?:\d+)(?:\.\d+)?(?:\.\d+)?`
	nameCv    = `[\w.-]+`

	ModuleTypeSrc ModuleType = iota
	ModuleTypeWheel
)

var (
	// /Flask_RESTful-0.3.5.dist-info
	// /Jinja2-2.9.6-py2.7.egg-info
	// /Django-1.6-py2.7.egg

	packageNameRe = regexp.MustCompile(
		fmt.Sprintf(`^(%s)-(%s)(?:.*)?(\.dist-info|\.egg-info|\.egg)$`, nameCv, versionCv),
	)
)

type (
	Repo struct {
		modules []*Pkg
	}

	Pkg struct {
		name       string
		rawVersion string
		version    versionarium.Version
		reqPath    string
		moduleType ModuleType
		lock       sync.Mutex
	}

	ModuleType int
)

func NewLocalRepo() (*Repo, error) {
	sitePackages := detectSitePackages()
	if len(sitePackages) == 0 {
		return nil, errors.New("failed site-packages folder")
	}

	repo := &Repo{
		modules: make([]*Pkg, 0),
	}
	repo.resolveModules(sitePackages)
	return repo, nil
}

func (r *Repo) FindPkg(depName string, dep manager.Dependency) (result *Pkg, err error) {
	depName = pypipkg.NormalizeName(depName)
	for _, module := range r.modules {
		if module.name != depName {
			continue
		}

		if !dep.CheckVersion(module.Version()) {
			simplelog.Debug("skip local version due to version check",
				"pkg_name", module.name,
				"pkg_version", module.rawVersion,
				"dep_version", dep.RawVersions,
			)
			continue
		}

		return module, nil
	}

	return
}

func (r *Repo) resolveModules(sitePackages []string) {
	for _, sitePath := range sitePackages {
		fileInfo, err := ioutil.ReadDir(sitePath)
		if err != nil {
			simplelog.Error("failed to list site-package", "path", sitePath, "err", err)
			continue
		}

		for _, file := range fileInfo {
			if !file.IsDir() {
				continue
			}

			pInfo := packageNameRe.FindStringSubmatch(file.Name())
			if len(pInfo) == 0 {
				continue
			}

			module := &Pkg{
				name:       pypipkg.NormalizeName(pInfo[1]),
				rawVersion: pInfo[2],
			}

			switch pInfo[3] {
			case ".egg-info":
				module.moduleType = ModuleTypeSrc
				module.reqPath = filepath.Join(sitePath, file.Name(), "requires.txt")
			case ".egg":
				module.moduleType = ModuleTypeSrc
				module.reqPath = filepath.Join(sitePath, file.Name(), "EGG-INFO", "requires.txt")
			case ".dist-info":
				module.moduleType = ModuleTypeWheel
				module.reqPath = filepath.Join(sitePath, file.Name(), "METADATA")
			default:
				// Unsupported
				simplelog.Info(fmt.Sprintf("Unsupported module info type: %s", pInfo[3]))
				continue
			}

			r.modules = append(r.modules, module)
		}
	}
}

func (m *Pkg) Version() versionarium.Version {
	m.lock.Lock()
	defer m.lock.Unlock()

	if m.version == nil {
		version, err := versionarium.NewVersion(lang, m.rawVersion)
		if err != nil {
			simplelog.Error("failed to parse local pkg version",
				"pkg_name", m.name,
				"pkg_path", m.reqPath,
				"err", err,
			)
			return nil
		}

		m.version = version
	}

	return m.version
}

func (m *Pkg) Requirements() (result *requirements.Requirements, resultErr error) {
	switch m.moduleType {
	case ModuleTypeSrc:
		requiresFile, err := os.Open(m.reqPath)
		if err != nil {
			break
		}

		requires, requiresExtras, err := pkgparser.ParseSrcRequires(requiresFile)
		_ = requiresFile.Close()
		if err != nil {
			simplelog.Error("failed to parse src requirements", "req_path", m.reqPath, "err", err)
			return
		}

		result, err = requirements.ParseLocal(requires, requiresExtras)
		if err != nil {
			simplelog.Error("failed to parse src requirements", "req_path", m.reqPath, "err", err)
			return
		}
	case ModuleTypeWheel:
		requiresFile, err := os.Open(m.reqPath)
		if err != nil {
			break
		}

		pkgInfo, err := pkgparser.ParseWheelMetadata(requiresFile)
		_ = requiresFile.Close()
		if err != nil {
			simplelog.Error("failed to parse src requirements", "req_path", m.reqPath, "err", err)
			return
		}

		result, err = requirements.ParseLocal(pkgInfo.Requires, pkgInfo.Extras)
		if err != nil {
			simplelog.Error("failed to parse src requirements", "req_path", m.reqPath, "err", err)
			return
		}
	default:
		// Unsupported
		simplelog.Error("unsupported local module type", "pkg_path", m.reqPath, "type", m.moduleType)
		return
	}

	return
}

func detectSitePackages() (sitePackages []string) {
	_, err := exec.LookPath("python")
	if err != nil {
		simplelog.Info("python interpreter not found")
		return
	}

	out, err := exec.Command("python", "-c", "import sys;import json; print(json.dumps(sys.path))").Output()
	if err != nil {
		simplelog.Error("failed to execute python helper: " + err.Error())
		return
	}

	var paths []string
	err = json.Unmarshal(out, &paths)
	if err != nil {
		return
	}

	for _, path := range paths {
		if strings.HasSuffix(path, "-packages") && fsutils.IsDirExists(path) {
			sitePackages = append(sitePackages, path)
		}
	}

	return
}
