package pip

import (
	"errors"
	"fmt"
	"path/filepath"
	"strings"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/versionarium"
	"a.yandex-team.ru/security/yadi/yadi/pkg/fsutils"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/pip/local"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/pip/remote"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/pip/requirements"
)

const (
	name = "pip"
	lang = "python"
)

var _ manager.PackageManager = (*Manager)(nil)

type (
	ManagerOpts struct {
		// Target file to parse
		TargetPath string

		// Dependency resolve mode
		// Default: local & remote
		ResolveMode manager.ResolveMode

		// Parse dev dependency
		WithDev bool

		// Don't use pypi cache
		// Default: false
		WithoutPypiCache bool
	}

	Manager struct {
		target      string
		withDev     bool
		resolveMode manager.ResolveMode
		remoteRepo  remote.Repo
		localRepo   *local.Repo
	}

	Module struct {
		Name        string
		RawVersion  string
		Version     versionarium.Version
		DownloadURI string
		Type        int
	}
)

func NewManager(opts ManagerOpts) (result *Manager, resultErr error) {
	if opts.TargetPath != "" && !fsutils.IsFileExists(opts.TargetPath) {
		resultErr = errors.New("target file not exists")
		return
	}

	if opts.ResolveMode == 0 {
		opts.ResolveMode = manager.ResolveLocal | manager.ResolveRemote
	}

	result = &Manager{
		target:      opts.TargetPath,
		withDev:     opts.WithDev,
		resolveMode: opts.ResolveMode,
	}

	if result.target == "" {
		result.resolveMode &^= manager.ResolveLocal
	} else if result.resolveMode&manager.ResolveLocal != 0 {
		var err error
		result.localRepo, err = local.NewLocalRepo()
		if err != nil {
			simplelog.Debug("disable local resolver disabled, but requested", "err", err)
			result.resolveMode &^= manager.ResolveLocal
		}
	}

	if result.resolveMode&manager.ResolveLocal == 0 && result.resolveMode&manager.ResolveRemote == 0 {
		return nil, errors.New("pip package manager works only as a Local or Remote resolver")
	}

	if opts.WithoutPypiCache {
		result.remoteRepo, resultErr = remote.NewSimpleRepo()
		if resultErr != nil {
			return
		}
	} else {
		result.remoteRepo, resultErr = remote.NewYadiRepo()
		if resultErr != nil {
			return
		}
	}

	return
}

func (m *Manager) Name() string {
	return name
}

func (m *Manager) Language() string {
	return lang
}

func (m *Manager) TargetPath() string {
	return m.target
}

func (m *Manager) Cacheable() bool {
	return true
}

func (m *Manager) CanLocal() bool {
	return m.resolveMode&manager.ResolveLocal != 0 && m.localRepo != nil
}

func (m *Manager) CanRemote() bool {
	return m.resolveMode&manager.ResolveRemote != 0
}

func (m *Manager) CanSuggest() bool {
	return true
}

func (m *Manager) RootModules() ([]manager.Module, error) {
	module, err := m.resolveModule(m.target)
	if err != nil {
		return nil, err
	}

	module.LocalPath = m.target
	return []manager.Module{module}, nil
}

func (m *Manager) ResolveRemoteDependency(dep manager.Dependency, parent manager.Module) (manager.Module, error) {
	simplelog.Debug("search dependency", "dependency", dep.FullName())
	name, extras := splitModuleName(dep.Name)
	moduleVersions, err := m.remoteRepo.FetchVersions(name)
	if err != nil {
		return manager.ZeroModule, err
	}

	var maxVer versionarium.Version
	var neededCandidate remote.Pkg
	for _, candidate := range moduleVersions {
		if !dep.CheckVersion(candidate.Version()) {
			continue
		}

		candidateVersion := candidate.Version()
		if maxVer != nil && maxVer.GreaterThan(candidateVersion) {
			continue
		}

		neededCandidate = candidate
		maxVer = candidateVersion
	}

	if neededCandidate == nil {
		return manager.ZeroModule, fmt.Errorf("failed to find dependency %s", dep.FullName())
	}

	module, err := moduleFromPypi(neededCandidate, extras)
	if err != nil {
		return manager.ZeroModule, err
	}

	return module, nil
}

func (m *Manager) ResolveLocalDependency(dep manager.Dependency, parent manager.Module) (manager.Module, error) {
	name, extras := splitModuleName(dep.Name)
	pkg, err := m.localRepo.FindPkg(name, dep)
	if err != nil {
		return manager.ZeroModule, err
	}

	if pkg == nil {
		return manager.ZeroModule, xerrors.Errorf("package %s not found", dep.Name)
	}

	moduleReqs, err := pkg.Requirements()
	if err != nil {
		return manager.ZeroModule, err
	}

	var dependencies []manager.Dependency
	if moduleReqs != nil {
		dependencies = moduleReqs.Dependencies(extras)
	}

	return manager.Module{
		Name:         dep.Name,
		Version:      pkg.Version(),
		Dependencies: dependencies,
	}, nil
}

func (m *Manager) SuggestModuleUpdate(module manager.Module, vulnerableVersions versionarium.VersionRange, parents []manager.Module) []string {
	currentVersions, err := m.remoteRepo.FetchVersions(module.Name)
	if err != nil {
		return nil
	}

	var parentVersions remote.PkgVersions
	var parent manager.Module
	// we always have "root" pkg, so we must to have more than 2 parents for transitive dep
	isDirect := len(parents) < 2
	if !isDirect {
		// Transitive dependency
		parent = parents[len(parents)-1]
		parentVersions, err = m.remoteRepo.FetchVersions(parent.Name)
		if err != nil {
			return nil
		}
	}

	for _, candidate := range currentVersions {
		if candidate.Version().ReleaseInfo() != "" {
			// Skip prerelease versions
			continue
		}

		if candidate.Version().LessThan(module.Version) {
			// Not needed
			continue
		}

		patchedVersion := candidate.Version()
		if vulnerableVersions.Check(patchedVersion) {
			// Vulnerable version
			continue
		}

		result := []string{fmt.Sprintf("%s@%s", module.Name, patchedVersion.String())}
		if isDirect {
			return result
		}

		for _, candidate := range parentVersions {
			if candidate.Version().ReleaseInfo() != "" {
				// Skip prerelease versions
				continue
			}

			if candidate.Version().LessThan(parent.Version) {
				// Not needed
				continue
			}

			parentModule, _ := moduleFromPypi(candidate, nil)
			for _, dep := range parentModule.Dependencies {
				if dep.Name != module.Name {
					// Not interesting dep
					continue
				}

				if !dep.CheckVersion(patchedVersion) {
					continue
				}

				parentVulnVersions, err := versionarium.NewRange(m.Language(), fmt.Sprintf("<%s", candidate.RawVersion()))
				if err != nil {
					simplelog.Error("failed to parse parent constraint",
						"module", candidate.Name(), "version", candidate.RawVersion(), "err", err)
					continue
				}

				parentVersion := m.SuggestModuleUpdate(parentModule, parentVulnVersions, parents[:len(parents)-1])
				if len(parentVersion) != 0 {
					return append(parentVersion, result...)
				}
			}
		}
	}

	return nil
}

func (m *Manager) resolveModule(path string) (manager.Module, error) {
	moduleReq, err := requirements.ReadRequirementsFile(path, m.withDev)
	if err != nil {
		return manager.ZeroModule, err
	}

	moduleName := filepath.Base(filepath.Dir(path))
	return manager.Module{
		Name:         moduleName,
		Dependencies: moduleReq.Dependencies(nil),
		Version:      versionarium.MustNewVersion(m.Language(), "0.0.0"),
	}, nil
}

func splitModuleName(moduleName string) (name string, extras []string) {
	index := strings.Index(moduleName, "[")
	if index == -1 {
		name = moduleName
		return
	}

	name = strings.TrimSpace(moduleName[0:index])
	rawExtras := strings.Split(moduleName[index+1:len(moduleName)-1], ",")
	for _, extra := range rawExtras {
		extras = append(extras, strings.TrimSpace(extra))
	}
	return
}

func moduleFromPypi(pkg remote.Pkg, extras []string) (manager.Module, error) {
	moduleReqs, err := pkg.Requirements()
	if err != nil {
		return manager.ZeroModule, err
	}

	moduleLicense, _ := pkg.License()
	return manager.Module{
		Name:         pkg.Name(),
		Version:      pkg.Version(),
		License:      moduleLicense,
		Dependencies: moduleReqs.Dependencies(extras),
	}, nil
}
