package remote

import (
	"encoding/json"
	"fmt"
	"net/url"
	"sync"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/yadi/libs/pypi"
	"a.yandex-team.ru/security/yadi/libs/versionarium"
	"a.yandex-team.ru/security/yadi/yadi/internal/config"
	"a.yandex-team.ru/security/yadi/yadi/internal/httputils"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/pip/requirements"
)

type (
	YadiPkg struct {
		name         string
		rawVersion   string
		version      versionarium.Version
		license      string
		fetched      bool
		fetchErr     error
		requirements *requirements.Requirements
		lock         sync.Mutex
	}

	remoteYadiPackageVersion struct {
		Ok     bool   `json:"ok"`
		Error  string `json:"error"`
		Result struct {
			Name         string                `json:"name"`
			Version      string                `json:"version"`
			License      string                `json:"license"`
			Requirements *pypi.PkgRequirements `json:"requirements"`
		} `json:"result"`
	}
)

func (p *YadiPkg) Name() string {
	return p.name
}

func (p *YadiPkg) RawVersion() string {
	return p.rawVersion
}

func (p *YadiPkg) Version() versionarium.Version {
	return p.version
}

func (p *YadiPkg) License() (string, error) {
	p.lock.Lock()
	defer p.lock.Unlock()

	if !p.fetched {
		p.fetch()
	}

	return p.license, p.fetchErr
}

func (p *YadiPkg) Requirements() (*requirements.Requirements, error) {
	p.lock.Lock()
	defer p.lock.Unlock()

	if !p.fetched {
		p.fetch()
	}

	return p.requirements, p.fetchErr
}

func (p *YadiPkg) fetch() {
	p.fetched = true

	uri := fmt.Sprintf("%s/api/v1/python/%s/%s", config.YadiHost, url.PathEscape(p.name), url.PathEscape(p.rawVersion))
	resp, err := httputils.DoGet(uri)
	if err != nil {
		p.fetchErr = xerrors.Errorf("failed to get yadi pkg: %w", err)
		return
	}
	defer httputils.GracefulClose(resp.Body)

	if resp.StatusCode != 200 {
		p.fetchErr = xerrors.Errorf("failed to get yadi pkg: non 200 status code: %d", resp.StatusCode)
		return
	}

	var pkg remoteYadiPackageVersion
	err = json.NewDecoder(resp.Body).Decode(&pkg)
	if err != nil {
		p.fetchErr = xerrors.Errorf("failed to decode yadi pkg: %w", err)
		return
	}

	if !pkg.Ok {
		p.fetchErr = xerrors.Errorf("failed to request yadi pkg: %s", pkg.Error)
		return
	}

	p.license = pkg.Result.License
	p.requirements, p.fetchErr = requirements.ParseRemote(pkg.Result.Requirements)
}
