package remote

import (
	"encoding/json"
	"fmt"
	"net/url"
	"sort"

	"github.com/karlseguin/ccache/v2"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/versionarium"
	"a.yandex-team.ru/security/yadi/yadi/internal/config"
	"a.yandex-team.ru/security/yadi/yadi/internal/httputils"
)

type (
	YadiRemoteRepo struct {
		versionsCache *ccache.Cache
	}

	remoteYadiPackage struct {
		Ok     bool   `json:"ok"`
		Error  string `json:"error"`
		Result struct {
			ID       uint64   `json:"id"`
			Name     string   `json:"name"`
			PkgName  string   `json:"pkg_name"`
			Source   string   `json:"source"`
			Versions []string `json:"versions"`
		} `json:"result"`
	}
)

func NewYadiRepo() (*YadiRemoteRepo, error) {
	return &YadiRemoteRepo{
		versionsCache: ccache.New(ccache.Configure().MaxSize(1000)),
	}, nil
}

func (s *YadiRemoteRepo) FetchVersions(name string) (PkgVersions, error) {
	versions, err := s.versionsCache.Fetch(name, cacheTTL, func() (interface{}, error) {
		uri := fmt.Sprintf("%s/api/v1/python/%s", config.YadiHost, url.PathEscape(name))
		resp, err := httputils.DoGet(uri)
		if err != nil {
			return nil, xerrors.Errorf("failed to get yadi pkg: %w", err)
		}
		defer httputils.GracefulClose(resp.Body)

		if resp.StatusCode != 200 {
			return nil, xerrors.Errorf("failed to get yadi pkg: non 200 status code: %d", resp.StatusCode)
		}

		var pkg remoteYadiPackage
		err = json.NewDecoder(resp.Body).Decode(&pkg)
		if err != nil {
			return nil, xerrors.Errorf("failed to decode yadi pkg: %w", err)
		}

		if !pkg.Ok {
			return nil, xerrors.Errorf("failed to request yadi pkg: %s", pkg.Error)
		}

		result := make(PkgVersions, 0, len(pkg.Result.Versions))
		for _, ver := range pkg.Result.Versions {
			clearedVersion := clearVersionRe.FindStringSubmatch(ver)
			if len(clearedVersion) == 0 {
				simplelog.Warn("failed to clear version", "version", ver)
				continue
			}

			parsedVersion, err := versionarium.NewVersion(lang, clearedVersion[1])
			if err != nil {
				simplelog.Warn("failed to parse version", "version", ver)
				continue
			}

			result = append(result, &YadiPkg{
				name:       pkg.Result.PkgName,
				rawVersion: ver,
				version:    parsedVersion,
			})
		}

		sort.Sort(result)
		return result, nil
	})

	if err != nil {
		return nil, err
	}

	return versions.Value().(PkgVersions), nil
}
