package pkglock

//go:generate easyjson

import (
	"fmt"
	"strings"

	"github.com/mailru/easyjson"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/versionarium"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager"
)

/*
Documentation: https://docs.npmjs.com/files/package-lock.json
Simplified example:
{
  "name": "test",
  "version": "0.0.0",
  "lockfileVersion": 1,
  "requires": true,
  "dependencies": {
    "express": {
      "version": "4.1.2",
      "requires": {
        "fresh": "0.2.2",
        "serve-static": "1.1.0"
      },
      "dependencies": {
        "serve-static": {
          "version": "1.1.0",
          "requires": {
            "parseurl": "1.0.1",
            "send": "0.3.0"
          }
        }
      }
    },
    "fresh": {
      "version": "0.2.2"
    }
  }
}
*/

const LockVersion = 2

type (
	//easyjson:json
	LockJSON struct {
		Name            string             `json:"name"`
		Version         string             `json:"version"`
		LockfileVersion int                `json:"lockfileVersion"`
		Dependencies    map[string]LockDep `json:"dependencies"`
	}

	//easyjson:json
	LockDep struct {
		Version      string             `json:"version"`
		From         string             `json:"from,omitempty"`
		Requires     map[string]string  `json:"requires,omitempty"`
		Dependencies map[string]LockDep `json:"dependencies,omitempty"`
		Dev          bool               `json:"dev"`
	}

	PackageLock struct {
		Root PackageLockModule
	}

	PackageLockDependencies map[string]PackageLockModule

	PackageLockRequirements map[string]string

	PackageLockModule struct {
		Name         string
		RawVersion   string
		Version      versionarium.Version
		Dependencies PackageLockDependencies
		Requires     PackageLockRequirements
		Parent       *PackageLockModule
		Dev          bool
	}
)

func (p *PackageLock) RootModule() manager.Module {
	// root module haven't any requirements, so we construct it manually :(
	if p.Root.Requires == nil {
		p.Root.Requires = make(PackageLockRequirements, len(p.Root.Dependencies))
	}

	for name, dep := range p.Root.Dependencies {
		p.Root.Requires[name] = dep.RawVersion
	}
	return p.Root.NewModule()
}

func (p *PackageLockModule) NewModule() manager.Module {
	return manager.Module{
		Name:         p.Name,
		Version:      p.Version,
		Dependencies: p.resolveDependencies(nil),
	}
}

func (p *PackageLockModule) resolveDependencies(parents []string) []manager.Dependency {
	reqCount := len(p.Requires)
	if reqCount == 0 {
		return nil
	}

	subParents := append(parents, p.Name)
	deps := make([]manager.Dependency, 0, reqCount)
	for name, rawVersions := range p.Requires {
		// TODO: optimize
		if isCircularReq(name, parents) {
			simplelog.Debug("circular recursion",
				"module", name, "path", strings.Join(parents, " -> "))
			continue
		}

		var dep PackageLockModule
		var ok bool
		if dep, ok = p.Dependencies[name]; !ok {
			for parent := p.Parent; parent != nil; parent = parent.Parent {
				if dep, ok = parent.Dependencies[name]; ok {
					break
				}
			}
		}

		if !ok {
			simplelog.Warn("failed to resolve requirements",
				"module", p.Name,
				"dependency", name,
				"versions", rawVersions,
			)
			continue
		}

		deps = append(deps, manager.Dependency{
			Name:                 name,
			ResolvedVersion:      dep.Version,
			RawVersions:          rawVersions,
			IsDev:                dep.Dev,
			ResolvedDependencies: dep.resolveDependencies(subParents),
			Language:             lang,
		})
	}
	return deps
}

func (p *PackageLockModule) String() string {
	if p.Version != nil {
		return fmt.Sprintf("%s@%s", p.Name, p.Version.String())
	}

	return p.Name
}

func (l *LockJSON) PackageLock(withDev bool) (*PackageLock, error) {
	ver, err := versionarium.NewVersion(lang, l.Version)
	if err != nil {
		simplelog.Warn("failed to parse root version", "version", l.Version, "err", err.Error())
		ver, _ = versionarium.NewVersion(lang, "0.0.0")
	}

	rootModule := PackageLockModule{
		Name:    l.Name,
		Version: ver,
	}

	deps, err := parseRawDependencies(l.Dependencies, &rootModule, withDev)
	if err != nil {
		return nil, err
	}

	rootModule.Dependencies = deps
	return &PackageLock{
		Root: rootModule,
	}, nil
}

func ParseLock(data []byte, withDev bool) (*PackageLock, error) {
	var lock LockJSON
	err := easyjson.Unmarshal(data, &lock)
	if err != nil {
		return nil, err
	}

	if lock.Name == "" {
		simplelog.Warn("no package name specified, using 'unspecified'")
		lock.Name = "unspecified"
	}

	if lock.LockfileVersion > LockVersion {
		return nil, fmt.Errorf("can't parse a package-lock.json of version %d, Yadi only supports up to %d",
			lock.LockfileVersion, LockVersion)
	}

	return lock.PackageLock(withDev)
}

func parseRawDependencies(deps map[string]LockDep, parent *PackageLockModule, withDev bool) (PackageLockDependencies, error) {
	if len(deps) == 0 {
		// Nothing to do
		return nil, nil
	}

	modules := make(PackageLockDependencies, len(deps))
	for name, dep := range deps {
		if !withDev && dep.Dev {
			// skip dev dependencies
			continue
		}

		if strings.HasPrefix(dep.Version, "http") || strings.HasPrefix(dep.Version, "git") {
			// skip invalid dep
			simplelog.Info("skip dependency", "name", name, "version", dep.Version)
			continue
		}

		ver, err := versionarium.NewVersion(lang, dep.Version)
		if err != nil {
			simplelog.Error("failed to parse version",
				"module", name,
				"version", ver,
				"err", err.Error(),
			)
			continue
		}

		module := PackageLockModule{
			Name:       name,
			Version:    ver,
			Parent:     parent,
			RawVersion: dep.Version,
			Requires:   dep.Requires,
			Dev:        dep.Dev,
		}

		subDeps, err := parseRawDependencies(dep.Dependencies, &module, withDev)
		if err != nil {
			return nil, err
		}

		module.Dependencies = subDeps
		modules[name] = module
	}

	return modules, nil
}

func isCircularReq(requirement string, parents []string) bool {
	for _, parent := range parents {
		if requirement == parent {
			return true
		}
	}

	return false
}
