package pkglock

import (
	"fmt"
	"io/ioutil"
	"path/filepath"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/versionarium"
	"a.yandex-team.ru/security/yadi/yadi/pkg/fsutils"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/npm"
)

const lang = "nodejs"

var _ manager.PackageManager = (*Manager)(nil)

type (
	ManagerOpts struct {
		// Target file to parse
		TargetPath string

		// Parse dev dependency
		WithDev bool
	}

	Manager struct {
		target  string
		withDev bool
		npm     *npm.Manager
		pkgLock *PackageLock
	}
)

func NewManager(opts ManagerOpts) (*Manager, error) {
	packageLock, err := readPackageLock(opts.TargetPath, opts.WithDev)
	if err != nil {
		return nil, err
	}

	npmManager, _ := npm.NewManager(npm.ManagerOpts{
		ResolveMode: manager.ResolveRemote,
		WithDev:     opts.WithDev,
	})

	return &Manager{
		target:  opts.TargetPath,
		withDev: opts.WithDev,
		npm:     npmManager,
		pkgLock: packageLock,
	}, nil
}

func (m *Manager) Name() string {
	return "pkglock"
}

func (m *Manager) Language() string {
	return lang
}

func (m *Manager) TargetPath() string {
	return m.target
}

func (m *Manager) Cacheable() bool {
	return false
}

func (m *Manager) CanLocal() bool {
	return true
}

func (m *Manager) CanRemote() bool {
	return false
}

func (m *Manager) CanSuggest() bool {
	return true
}

func (m *Manager) RootModules() ([]manager.Module, error) {
	if m.target != "" {
		packagePath := filepath.Join(filepath.Dir(m.target), "package.json")
		if fsutils.IsFileExists(packagePath) {
			return m.npm.ResolveRootWithSubmodules(packagePath)
		}

		simplelog.Warn("failed to find 'package.json', Yadi will analyze flatten package-lock.json dependencies")
	}

	return []manager.Module{m.pkgLock.RootModule()}, nil
}

func (m *Manager) ResolveLocalDependency(dep manager.Dependency, _ manager.Module) (manager.Module, error) {
	if dep.ResolvedVersion == nil {
		// probably this is root module
		if module, ok := m.pkgLock.Root.Dependencies[dep.Name]; ok {
			return module.NewModule(), nil
		}
		return manager.ZeroModule, fmt.Errorf("failed to resolve local module '%s': empty resolved version", dep.Name)
	}

	// Otherwise this is already resolved dependency
	return manager.Module{
		Name:         dep.Name,
		Version:      dep.ResolvedVersion,
		Dependencies: dep.ResolvedDependencies,
	}, nil
}

func (m *Manager) ResolveRemoteDependency(_ manager.Dependency, _ manager.Module) (manager.Module, error) {
	panic("not implemented: must not be called")
}

func (m *Manager) SuggestModuleUpdate(module manager.Module, vulnerableVersions versionarium.VersionRange, parents []manager.Module) []string {
	return m.npm.SuggestModuleUpdate(module, vulnerableVersions, parents)
}

func readPackageLock(path string, withDev bool) (*PackageLock, error) {
	if !fsutils.IsFileExists(path) {
		return nil, xerrors.Errorf("file not exists: %s", path)
	}

	data, err := ioutil.ReadFile(path)
	if err != nil {
		return nil, err
	}

	result, err := ParseLock(data, withDev)
	if err != nil {
		return nil, err
	}
	return result, nil
}
