package yarn

import (
	"errors"
	"fmt"
	"io/ioutil"
	"path"
	"path/filepath"
	"strings"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/versionarium"
	"a.yandex-team.ru/security/yadi/yadi/pkg/fsutils"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/npm"
)

const lang = "nodejs"

var _ manager.PackageManager = (*Manager)(nil)

type (
	ManagerOpts struct {
		// Target file to parse
		TargetPath string

		// Parse dev dependency
		WithDev bool
	}

	Manager struct {
		target   string
		withDev  bool
		npm      *npm.Manager
		yarnLock Lock
		cwd      string
	}
)

func NewManager(opts ManagerOpts) (*Manager, error) {
	if opts.TargetPath != "" && !fsutils.IsFileExists(opts.TargetPath) {
		return nil, errors.New("target file not exists")
	}

	yarnLock, err := readYarn(opts.TargetPath)
	if err != nil {
		return nil, err
	}

	npmManager, _ := npm.NewManager(npm.ManagerOpts{
		ResolveMode: manager.ResolveRemote,
		WithDev:     opts.WithDev,
	})

	return &Manager{
		target:   opts.TargetPath,
		withDev:  opts.WithDev,
		npm:      npmManager,
		yarnLock: yarnLock,
		cwd:      path.Dir(opts.TargetPath),
	}, nil
}

func (m *Manager) Name() string {
	return "yarn"
}

func (m *Manager) Language() string {
	return lang
}

func (m *Manager) TargetPath() string {
	return m.target
}

func (m *Manager) CanLocal() bool {
	return true
}

func (m *Manager) Cacheable() bool {
	return false
}

func (m *Manager) CanRemote() bool {
	return false
}

func (m *Manager) CanSuggest() bool {
	return true
}

func (m *Manager) RootModules() ([]manager.Module, error) {
	packagePath := filepath.Join(filepath.Dir(m.target), "package.json")
	if fsutils.IsFileExists(packagePath) {
		return m.npm.ResolveRootWithSubmodules(packagePath)
	}

	simplelog.Warn("failed to find 'package.json', Yadi will analyze flatten Yarn dependencies")
	deps := make([]manager.Dependency, 0)
	for from := range m.yarnLock {
		name, rawVersions := splitFrom(from)

		deps = append(deps, manager.Dependency{
			Name:        name,
			RawVersions: rawVersions,
		})
	}

	ver, _ := versionarium.NewVersion(lang, "0.0.0")
	return []manager.Module{{
		Name:         filepath.Base(filepath.Dir(m.target)),
		Version:      ver,
		Dependencies: deps,
	}}, nil
}

func (m *Manager) ResolveLocalDependency(dep manager.Dependency, _ manager.Module) (manager.Module, error) {
	key := dep.FullName()
	if dep.IsStrictlyLocal() && strings.HasPrefix(dep.LocalPath, m.cwd) {
		// Rebuild local file key
		// TODO(buglloc): too ugly!
		key = strings.TrimPrefix(dep.LocalPath, m.cwd+"/")
		key = strings.TrimSuffix(key, "/package.json")
		key = fmt.Sprintf("%s@file:%s", dep.Name, key)
	}

	if ymodule, ok := m.yarnLock[key]; ok {
		module := ymodule.NewModule()
		return module, nil
	}
	return manager.ZeroModule, fmt.Errorf("module %q not found", dep.FullName())
}

func (m *Manager) ResolveRemoteDependency(_ manager.Dependency, _ manager.Module) (manager.Module, error) {
	panic("not implemented: must not be called")
}

func (m *Manager) SuggestModuleUpdate(module manager.Module, vulnerableVersions versionarium.VersionRange, parents []manager.Module) []string {
	return m.npm.SuggestModuleUpdate(module, vulnerableVersions, parents)
}

func readYarn(path string) (Lock, error) {
	data, err := ioutil.ReadFile(path)
	if err != nil {
		return nil, err
	}

	result, err := ParseYarnLock(data)
	if err != nil {
		return nil, err
	}
	return result, nil
}
