package yarn

import (
	"strings"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/versionarium"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/yarn/yarnparser"
)

type (
	/*
	   "@protobufjs/fetch@^1.1.0":
	     version "1.1.0"
	     resolved "https://registry.yarninternal.com/@protobufjs/fetch/-/fetch-1.1.0.tgz#ba99fb598614af65700c1619ff06d454b0d84c45"
	     dependencies:
	       "@protobufjs/aspromise" "^1.1.1"
	       "@protobufjs/inquire" "^1.1.0"
	*/

	Module struct {
		Name         string
		RawVersion   string
		Version      versionarium.Version
		Dependencies []manager.Dependency
	}

	Lock map[string]Module
)

func (y *Module) NewModule() manager.Module {
	return manager.Module{
		Name:         y.Name,
		Version:      y.Version,
		Dependencies: y.Dependencies,
	}
}

func ParseYarnLock(data []byte) (Lock, error) {
	parser, err := yarnparser.NewParser(data)
	if err != nil {
		return nil, err
	}

	yarnLock, err := parser.Parse()
	if err != nil {
		return nil, err
	}

	result := make(Lock)
	for from, info := range yarnLock {
		moduleInfo := info.(yarnparser.YarnValue)
		name, _ := splitFrom(from)
		rawVersion, ok := moduleInfo["version"].(string)
		if !ok {
			simplelog.Error("version not found for: " + from)
			continue
		}

		var version versionarium.Version
		var err error
		if isAcceptableVersion(rawVersion) {
			version, err = versionarium.NewVersion(lang, rawVersion)
			if err != nil {
				simplelog.Error("failed to parse version: "+err.Error(), "module", name, "version", rawVersion)
				continue
			}
		}

		dependencies := make([]manager.Dependency, 0)
		if deps, ok := moduleInfo["dependencies"]; ok {
			dependencies = appendDependencies(dependencies, deps.(yarnparser.YarnValue), false)
		}

		if deps, ok := moduleInfo["optionalDependencies"]; ok {
			dependencies = appendDependencies(dependencies, deps.(yarnparser.YarnValue), true)
		}

		result[from] = Module{
			Name:         name,
			RawVersion:   rawVersion,
			Version:      version,
			Dependencies: dependencies,
		}
	}

	return result, nil
}

func appendDependencies(target []manager.Dependency, source yarnparser.YarnValue, isDev bool) []manager.Dependency {
	for dname, dver := range source {
		rawVersions := dver.(string)
		if !isAcceptableVersion(rawVersions) {
			continue
		}

		target = append(target, manager.Dependency{
			Name:        dname,
			IsDev:       isDev,
			RawVersions: rawVersions,
			Language:    lang,
		})
	}
	return target
}

func splitFrom(from string) (string, string) {
	if i := strings.LastIndex(from, "@"); i != -1 {
		return from[:i], from[i+1:]
	}
	return from, ""
}

func isAcceptableVersion(version string) bool {
	return !strings.HasPrefix(version, "http") && !strings.HasPrefix(version, "git") && !strings.HasPrefix(version, "file")
}
