package yarnparser

import (
	"errors"
	"fmt"
	"regexp"
	"strings"

	"a.yandex-team.ru/security/libs/go/simplelog"
)

const (
	LockfileVersion = "1"
)

type (
	YarnValue map[string]interface{}

	Parser struct {
		token   *Token
		tokens  []*Token
		current int
	}
)

var (
	versionRe = regexp.MustCompile(`^yarn lockfile v(\d+)$`)
)

func NewParser(data []byte) (*Parser, error) {
	tokens, err := NewTokenizer().Tokenize(data)
	if err != nil {
		return nil, err
	}

	return &Parser{
		tokens:  tokens,
		current: 0,
	}, nil
}

func (p *Parser) Parse() (YarnValue, error) {
	p.next()
	return p.parse(0)
}

func (p *Parser) next() *Token {
	if p.current >= len(p.tokens) {
		simplelog.Error("yarn parser: no more tokens")
		return nil
	}

	val := p.tokens[p.current]
	p.current++
	if val.Type == COMMENT {
		p.onComment(val)
		return p.next()
	}

	p.token = val
	return p.token
}

func (p *Parser) parse(indent int) (YarnValue, error) {
	result := make(YarnValue)
Loop:
	for {
		token := p.token
		switch token.Type {
		case EOF:
			break Loop
		case NEWLINE:
			nextToken := p.next()
			if indent == 0 {
				// if we have 0 indentation then the next token doesn't matter
				continue
			}

			if nextToken.Type != INDENT {
				// if we have no indentation after a newline then we've gone down a level
				break Loop
			}

			if nextToken.Int() == indent {
				// all is good, the indent is on our level
				p.next()
			} else {
				// the indentation is less than our level
				break Loop
			}
		case INDENT:
			if token.Int() == indent {
				p.next()
			} else {
				break Loop
			}
		case STRING:
			keys := make([]string, 0)
			keys = append(keys, token.String())
			p.next()

			// support multiple keys
			for p.token.Type == COMMA {
				keyToken := p.next() // skip comma
				if keyToken.Type != STRING {
					return nil, errors.New("expected string followed by comma")
				}

				keys = append(keys, keyToken.String())
				p.next()
			}

			valToken := p.token
			if valToken.Type == COLON {
				// object
				p.next()

				// parse object
				val, err := p.parse(indent + 1)
				if err != nil {
					return nil, err
				}

				for _, key := range keys {
					result[key] = val
				}

				if indent > 0 && p.token.Type != INDENT {
					break Loop
				}
			} else if valToken.Type == BOOLEAN || valToken.Type == STRING || valToken.Type == NUMBER {
				// plain value
				for _, key := range keys {
					result[key] = valToken.Val
				}
				p.next()
			} else {
				return nil, fmt.Errorf("invalid value type: %s (%d:%d)",
					TokenTypeString(valToken.Type), valToken.Line, valToken.Col)
			}
		default:
			return nil, fmt.Errorf("unexpected token: %s (%d:%d)",
				TokenTypeString(token.Type), token.Line, token.Col)
		}
	}
	return result, nil
}

func (p *Parser) onComment(token *Token) {
	comment := strings.TrimSpace(token.String())
	versionMatch := versionRe.FindStringSubmatch(comment)
	if len(versionMatch) > 0 {
		version := versionMatch[1]
		if version > LockfileVersion {
			simplelog.Error(fmt.Sprintf("can't parse a lockfile of version %s, Yadi only supports up to %s",
				version, LockfileVersion))
		}
	}
}
