package yarnaudit

import (
	"fmt"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/yadi/libs/versionarium"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/npm"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/yarn"
)

const (
	name = "yarnaudit"
	lang = "nodejs"
)

var _ manager.PackageManager = (*Manager)(nil)

type (
	ManagerOpts struct {
		// Target PkgJSON
		PkgJSON *npm.PackageJSON

		// Target PkgLock
		YarnLock []byte

		// Parse dev dependency
		WithDev bool
	}

	Manager struct {
		withDev  bool
		pkgJSON  *npm.PackageJSON
		yarnLock yarn.Lock
	}
)

func NewManager(opts ManagerOpts) (*Manager, error) {
	yarnLock, err := yarn.ParseYarnLock(opts.YarnLock)
	if err != nil {
		return nil, xerrors.Errorf("failed to parse yarn.lock: %w", err)
	}

	return &Manager{
		withDev:  opts.WithDev,
		pkgJSON:  opts.PkgJSON,
		yarnLock: yarnLock,
	}, nil
}

func (m *Manager) Name() string {
	return name
}

func (m *Manager) Language() string {
	return lang
}

func (m *Manager) TargetPath() string {
	return "n/a"
}

func (m *Manager) Cacheable() bool {
	return false
}

func (m *Manager) CanLocal() bool {
	return true
}

func (m *Manager) CanRemote() bool {
	return false
}

func (m *Manager) CanSuggest() bool {
	return false
}

func (m *Manager) RootModules() ([]manager.Module, error) {
	return []manager.Module{
		m.pkgJSON.NewModule(m.withDev),
	}, nil
}

func (m *Manager) ResolveLocalDependency(dep manager.Dependency, _ manager.Module) (manager.Module, error) {
	key := dep.FullName()
	if ymodule, ok := m.yarnLock[key]; ok {
		module := ymodule.NewModule()
		return module, nil
	}

	return manager.ZeroModule, fmt.Errorf("module %q not found", dep.FullName())
}

func (m *Manager) ResolveRemoteDependency(dep manager.Dependency, _ manager.Module) (manager.Module, error) {
	panic("not implemented: must not be called")
}

func (m *Manager) SuggestModuleUpdate(_ manager.Module, _ versionarium.VersionRange, _ []manager.Module) []string {
	return nil
}
