package consoleout

import (
	"bytes"
	"os"
	"strings"
	"text/template"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/yadi/internal/results"
	"a.yandex-team.ru/security/yadi/yadi/pkg/outputs/commonout"
	"a.yandex-team.ru/security/yadi/yadi/pkg/outputs/outer"
)

var _ outer.IssueOutput = (*IssueOutput)(nil)

const issueTmpl = `
{{- $cdefault := "\033[0m" -}}
{{$ctitle := "\033[95m" -}}
{{$chightlight := "\033[1m" -}}
{{range $i, $result := . -}}
{{if $result.Path}}{{ $ctitle }}{{ $result.Path }}{{ $cdefault }}:{{end}}
{{range $i, $issue := $result.Issues}}
  {{ severityColor $issue.Severity }}{{ $issue.Severity }} severity vulnerability found on {{ $issue.String }}{{ $cdefault }}:
    - desc: {{ $issue.Summary }}
    - id: {{ $issue.ID }}
    - info: {{ $issue.Reference }}
    - path: {{ join $issue.Path " > " }}
    - patched{{if ne $issue.PatchedVersions ""}} in: {{ $issue.PatchedVersions }}{{ else if $issue.PatchExists }}: yes{{ else }}: n/a{{ end }}
    - vulnerable versions: {{ $issue.RawVersions }}
	{{if $issue.Suggestable}}{{if $issue.Suggest}}{{ $chightlight }}{{end}}{{ suggest $issue }}{{if $issue.Suggest}}{{ $cdefault }}{{end}}{{end}}
{{end -}}
{{- end -}}
`

const summaryTmpl = `
{{$cdefault := "\033[0m" -}}
{{ severityColor .MaxSeverity }}Found {{ .Issues}} vulnerabilities{{ $cdefault }} in {{ .Path}} vulnerable paths
`

type IssueOutput struct {
	issueTmpl   *template.Template
	summaryTmpl *template.Template
}

func NewIssueOutput() *IssueOutput {
	funs := template.FuncMap{
		"join":          strings.Join,
		"severityColor": severityToConsoleColor,
		"suggest":       commonout.FormatSuggest,
	}
	return &IssueOutput{
		issueTmpl: template.Must(
			template.New("console_result").
				Funcs(funs).
				Parse(issueTmpl),
		),
		summaryTmpl: template.Must(
			template.New("console_summary").
				Funcs(funs).
				Parse(summaryTmpl),
		),
	}
}

func (o *IssueOutput) WriteIssues(results []results.Analyze) {
	if len(results) == 0 {
		return
	}

	var result bytes.Buffer
	err := o.issueTmpl.Execute(&result, results)
	if err != nil {
		simplelog.Error("failed to render results", "err", err)
	}

	_ = o.summaryTmpl.Execute(&result, commonout.CalculateSummary(results))
	_, _ = os.Stdout.Write(result.Bytes())
}

func (o *IssueOutput) Close() error {
	return nil
}

func severityToConsoleColor(severity string) string {
	switch severity {
	case "Low":
		return "\033[94m"
	case "Medium":
		return "\033[93m"
	case "High":
		return "\033[91m"
	case "Critical":
		return "\033[31;1m"
	default:
		// info level
		return "\033[0m"
	}
}
