package genericout

import (
	"encoding/json"
	"fmt"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/yadi/pkg/analyze"
	"a.yandex-team.ru/security/yadi/yadi/pkg/fsutils"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager"
	"a.yandex-team.ru/security/yadi/yadi/pkg/outputs/outer"
	"a.yandex-team.ru/security/yadi/yadi/pkg/outputs/writer"
)

var _ outer.ListOutput = (*ListOutput)(nil)

type ListOutput struct {
	lvl      int
	deps     []bool
	writer   *writer.Writer
	prjName  string
	language string
}

type Module struct {
	PackageName  string   `json:"package_name"`
	Version      string   `json:"version"`
	Path         string   `json:"path,omitempty"`
	License      string   `json:"license,omitempty"`
	Language     string   `json:"language,omitempty"`
	From         string   `json:"from,omitempty"`
	Dependencies []Module `json:"dependencies,omitempty"`
}
type ListOption func(*ListOutput)

func WithWriter(w *writer.Writer) ListOption {
	return func(o *ListOutput) {
		o.writer = w
	}
}

func WithProjectName(name string) ListOption {
	return func(o *ListOutput) {
		o.prjName = name
	}
}

func WithLanguage(language string) ListOption {
	return func(o *ListOutput) {
		o.language = language
	}
}

func NewListOutput(opts ...ListOption) *ListOutput {
	o := &ListOutput{
		prjName: fsutils.CurrentDir(),
		lvl:     -1,
		writer:  writer.NewWriter(),
	}

	for _, opt := range opts {
		opt(o)
	}

	_, _ = o.writer.WriteString(fmt.Sprintf(`{"project_name": "%s", "language": "%s", "root_modules":[`, o.prjName, o.language))
	return o
}

func (o *ListOutput) EnterRootModule(module manager.Module, pm manager.PackageManager) {
	if o.lvl >= 0 {
		_ = o.writer.WriteByte(',')
	}

	o.lvl = 0
	o.deps = append(o.deps[:0], false)

	tp := pm.TargetPath()

	m := Module{
		Path:        tp,
		Language:    pm.Language(),
		PackageName: module.Name,
		License:     module.License,
		Version:     module.Version.String(),
	}
	out, err := json.Marshal(m)
	if err != nil {
		simplelog.Error("failed to marshal root module", "module", module.Name)
		return
	}

	_, _ = o.writer.Write(out[:len(out)-1])
	_, _ = o.writer.WriteString(`,"dependencies":[`)
}

func (o *ListOutput) ExitRootModule(_ *analyze.DependencyStats) {
	_, _ = o.writer.WriteString("]}")
}

func (o *ListOutput) EnterModule(module manager.Module, from string, _ []manager.Module) {
	m := Module{
		PackageName: module.Name,
		License:     module.License,
		Version:     module.Version.String(),
		From:        from,
	}
	out, err := json.Marshal(m)
	if err != nil {
		simplelog.Error("failed to marshal root module", "module", module.Name)
		return
	}

	if !o.deps[o.lvl] {
		o.deps[o.lvl] = true
	} else {
		_ = o.writer.WriteByte(',')
	}

	o.lvl++
	o.deps = append(o.deps, false)
	_, _ = o.writer.Write(out[:len(out)-1])
	_, _ = o.writer.WriteString(`,"dependencies":[`)
}

func (o *ListOutput) ExitModule() {
	o.deps = o.deps[:o.lvl]
	o.lvl--
	_, _ = o.writer.WriteString("]}")
}

func (o *ListOutput) Close() error {
	_, _ = o.writer.WriteString("]}")
	return o.writer.Close()
}
