package startrekout

import (
	"bytes"
	"fmt"
	"os"
	"strings"
	"text/template"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/yadi/internal/config"
	"a.yandex-team.ru/security/yadi/yadi/internal/results"
	"a.yandex-team.ru/security/yadi/yadi/pkg/analyze"
	"a.yandex-team.ru/security/yadi/yadi/pkg/outputs/outer"
)

var _ outer.IssueOutput = (*IssueOutput)(nil)

const (
	tmpl = `
Обнаружено использование компонентов с известными уязвимостями.
{{range $i, $result := . -}}
{{ $result.Path }}:
{{range $i, $issue := $result.Issues}}
  {{ $col := severityColor $issue.Severity -}}
  * **{{if $col}}!!({{ severityColor $issue.Severity }}){{end}}{{ $issue.String }}: {{ $issue.Summary }}{{if $col}}!!{{end}}**:
  {{if (ne (len $issue.Path) 1)}}Используется в цепочке %%{{ join $issue.Path "%% > %%" }}%%{{end}}
  Починено{{if ne $issue.PatchedVersions ""}} в версиях: {{ $issue.PatchedVersions }}{{ else if $issue.PatchExists }}: да{{ else }}: нет{{ end }}
  {{if $issue.Suggestable}}{{if $issue.Suggest}}Для починки можно {{ suggest $issue }}{{end}}{{end}}
  Подробнее в {{ $issue.Reference }}
{{end}}
{{end -}}
`
	infoSeverity     = "Info"
	lowSeverity      = "Low"
	mediumSeverity   = "Medium"
	highSeverity     = "High"
	criticalSeverity = "Critical"

	grayColor = "gray"
	redColor  = "red"
)

type IssueOutput struct {
	tmpl *template.Template
}

func NewIssueOutput() *IssueOutput {
	return &IssueOutput{
		tmpl: template.Must(
			template.New("startreck_result").
				Funcs(template.FuncMap{
					"join":          strings.Join,
					"suggest":       formatSuggestSt,
					"severityColor": severityToStColor,
					"stats":         calculateStats,
				}).
				Parse(tmpl),
		),
	}
}

func (o *IssueOutput) WriteIssues(results []results.Analyze) {
	if len(results) == 0 {
		return
	}

	var result bytes.Buffer
	err := o.tmpl.Execute(&result, results)
	if err != nil {
		simplelog.Error("failed to render results", "err", err)
	}

	_, _ = os.Stdout.Write(result.Bytes())
}

func (o *IssueOutput) Close() error {
	return nil
}

func severityToStColor(severity string) (color string) {
	switch severity {
	case infoSeverity:
		return grayColor
	case lowSeverity:
		return grayColor
	case mediumSeverity:
		return
	case highSeverity:
		return redColor
	case criticalSeverity:
		return redColor
	}
	return
}

func calculateStats(results analyze.IssueList) string {
	info := 0
	low := 0
	medium := 0
	high := 0
	critical := 0

	for _, issue := range results {
		switch issue.Severity() {
		case infoSeverity:
			info++
		case lowSeverity:
			low++
		case mediumSeverity:
			medium++
		case highSeverity:
			high++
		case criticalSeverity:
			critical++
		}
	}

	result := make([]string, 0)
	if critical > 0 {
		result = append(result, fmt.Sprintf("Critical: %d", critical))
	}

	if high > 0 {
		result = append(result, fmt.Sprintf("High: %d", high))
	}

	if medium > 0 {
		result = append(result, fmt.Sprintf("Medium: %d", medium))
	}

	if low > 0 {
		result = append(result, fmt.Sprintf("Low: %d", low))
	}

	if info > 0 {
		result = append(result, fmt.Sprintf("Info: %d", info))
	}

	return strings.Join(result, ", ")
}

func formatSuggestSt(issue analyze.Issue) string {
	if !config.SuggestUpdate {
		return ""
	}

	if len(issue.Suggest) == 0 {
		return ""
	}

	suggests := issue.Suggest
	var suggest string
	if issue.Path[0] == suggests[0] {
		suggest = "переустановить  %%" + issue.Path[0] + "%%"
	} else {
		suggest = "обновить %%" + issue.Path[0] + "%% до %%" + suggests[0] + "%%"
	}

	if len(suggests) > 1 {
		suggest += fmt.Sprintf(" (повлечет обновление %%%%%s%%%%)", strings.Join(suggests[1:], "%% > %%"))
	}
	return suggest
}
