package teamcityout

import (
	"bytes"
	"fmt"
	"os"
	"strings"

	"a.yandex-team.ru/security/yadi/yadi/internal/results"
	"a.yandex-team.ru/security/yadi/yadi/pkg/analyze"
	"a.yandex-team.ru/security/yadi/yadi/pkg/outputs/commonout"
	"a.yandex-team.ru/security/yadi/yadi/pkg/outputs/outer"
)

var _ outer.IssueOutput = (*IssueOutput)(nil)

type IssueOutput struct {
}

func NewIssueOutput() *IssueOutput {
	return &IssueOutput{}
}

/*
Example:
##teamcity[testSuiteStarted name='yadi']
##teamcity[testSuiteStarted name='test']
##teamcity[testStarted name='requests']
##teamcity[testFailed name='requests' message='Medium severity vulnerability found on requests@2.1.0' details='- file: examples/pip-transitive/requirements.txt|n- desc: Information Disclosure|n- id: sc:2048:98779946|n- info: https://yadi.yandex-team.ru/vulns/vuln/sc:2048:98779946|n- patched: >=2.3.0|n- path: chef_gae@1.0.3 > requests@2.1.0|nNo direct dependency upgrade can address this issue.|n|n']
##teamcity[testFinished name='requests']
##teamcity[testSuiteFinished name='test']
##teamcity[testSuiteFinished name='yadi']
*/
func (o *IssueOutput) WriteIssues(results []results.Analyze) {
	if len(results) == 0 {
		return
	}

	var out bytes.Buffer
	out.WriteString("##teamcity[testSuiteStarted name='yadi']\n")
	out.WriteString("##teamcity[testSuiteStarted name='test']\n")
	for _, result := range results {
		for _, issue := range result.Issues {
			name := getTestname(issue)
			out.WriteString(fmt.Sprintf("##teamcity[testStarted name='%s']\n", name))
			out.WriteString(fmt.Sprintf("##teamcity[testFailed name='%s' message='%s' details='%s']\n",
				name, getMessage(issue), getDetails(issue, result.Path)))
			out.WriteString(fmt.Sprintf("##teamcity[testFinished name='%s']\n", name))
		}
	}

	out.WriteString("##teamcity[testSuiteFinished name='test']\n")
	out.WriteString("##teamcity[testSuiteFinished name='yadi']\n")
	_, _ = os.Stdout.Write(out.Bytes())
}

func (o *IssueOutput) Close() error {
	return nil
}

func teamcityEscape(data string) string {
	var result []rune
	for _, c := range data {
		switch c {
		case '\'':
			result = append(result, '|', '\'')
		case '\n':
			result = append(result, '|', 'n')
		case '\r':
			result = append(result, '|', 'r')
		case '|':
			result = append(result, '|', '|')
		case ']':
			result = append(result, '|', ']')
		default:
			result = append(result, c)
		}
	}

	return string(result)
}

func getTestname(issue analyze.Issue) string {
	return teamcityEscape(issue.Path[0])
}

func getMessage(issue analyze.Issue) string {
	result := fmt.Sprintf("%s severity vulnerability found on %s", issue.Severity(), issue.String())
	return teamcityEscape(result)
}

func getDetails(issue analyze.Issue, path string) string {
	var out strings.Builder
	_, _ = fmt.Fprintf(&out, "- root: %s\n", path)
	_, _ = fmt.Fprintf(&out, "- desc: %s\n", issue.Summary)
	_, _ = fmt.Fprintf(&out, "- id: %s\n", issue.ID)
	_, _ = fmt.Fprintf(&out, "- info: %s\n", issue.Reference)
	if issue.PatchedVersions != "" {
		_, _ = fmt.Fprintf(&out, "- patched: %s\n", issue.PatchedVersions)
	} else if issue.PatchExists {
		out.WriteString("- patched: yes\n")
	} else {
		out.WriteString("- patched: n/a\n")
	}
	_, _ = fmt.Fprintf(&out, "- path: %s\n", strings.Join(issue.Path, " > "))
	if issue.Suggestable {
		out.WriteString(commonout.FormatSuggest(issue))
	}

	out.WriteString("\n\n")
	return teamcityEscape(out.String())
}
