package textout

import (
	"bytes"
	"os"
	"strings"
	"text/template"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/yadi/internal/results"
	"a.yandex-team.ru/security/yadi/yadi/pkg/outputs/commonout"
	"a.yandex-team.ru/security/yadi/yadi/pkg/outputs/outer"
)

var _ outer.IssueOutput = (*IssueOutput)(nil)

const issueTmpl = `
{{- range $i, $result := . -}}
{{if $result.Path}}{{ $result.Path }}:{{end}}
{{range $i, $issue := $result.Issues}}
  {{ $issue.Severity }} severity vulnerability found on {{ $issue.String }}:
    - desc: {{ $issue.Summary }}
    - id: {{ $issue.ID }}
    - info: {{ $issue.Reference }}
    - patched{{if ne $issue.PatchedVersions ""}} in: {{ $issue.PatchedVersions }}{{ else if $issue.PatchExists }}: yes{{ else }}: n/a{{ end }}
    - vulnerable versions: {{ $issue.RawVersions }}
    - path: {{ join $issue.Path " > " }}
	{{if $issue.Suggestable}}{{ suggest $issue }}{{end}}
{{end -}}
{{- end -}}
`

const summaryTmpl = `
Found {{ .Issues}} vulnerabilities in {{ .Path}} vulnerable paths
`

type IssueOutput struct {
	issueTmpl   *template.Template
	summaryTmpl *template.Template
}

func NewIssueOutput() *IssueOutput {
	return &IssueOutput{
		issueTmpl: template.Must(
			template.New("text_result").
				Funcs(template.FuncMap{
					"join":    strings.Join,
					"suggest": commonout.FormatSuggest,
				}).
				Parse(issueTmpl),
		),
		summaryTmpl: template.Must(template.New("text_summary").Parse(summaryTmpl)),
	}
}

func (o *IssueOutput) WriteIssues(results []results.Analyze) {
	if len(results) == 0 {
		return
	}

	var result bytes.Buffer
	err := o.issueTmpl.Execute(&result, results)
	if err != nil {
		simplelog.Error("failed to render results", "err", err)
	}

	_ = o.summaryTmpl.Execute(&result, commonout.CalculateSummary(results))
	_, _ = os.Stdout.Write(result.Bytes())
}

func (o *IssueOutput) Close() error {
	return nil
}
