package main

import (
	"context"
	"os"
	"os/signal"
	"syscall"
	"time"

	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/yaudit/internal/config"
	"a.yandex-team.ru/security/yadi/yaudit/internal/server"
)

const (
	shutdownDeadline = time.Second * 30
)

func nonEmpty(sources ...string) string {
	for _, str := range sources {
		if str != "" {
			return str
		}
	}
	panic("empty strings")
}

func newConfig() config.Config {
	var cfg config.Config
	switch os.Getenv("YAUDIT_INSTALLATION") {
	case "dev":
		simplelog.SetLevel(simplelog.DebugLevel)
		cfg = config.Config{
			Debug:           true,
			HTTPPort:        3000,
			FeedPath:        nonEmpty(os.Getenv("VULNFEED_PATH"), "https://s3.mdst.yandex.net/yaditestdb/{lang}.json.gz"),
			ManifestPath:    nonEmpty(os.Getenv("MANIFEST_PATH"), "https://s3.mdst.yandex.net/yaditestdb/manifest.json"),
			WithDev:         false,
			RequestTimeout:  5,
			WatcherCronSpec: nonEmpty(os.Getenv("FEED_UPDATE_CRON_SPEC"), "0/1 * * * *"),
			HecSenderToken:  os.Getenv("YAUDIT_HEC_TOKEN"),
		}

	case "testing":
		simplelog.SetLevel(simplelog.InfoLevel)
		cfg = config.Config{
			Debug:           false,
			HTTPPort:        80,
			FeedPath:        nonEmpty(os.Getenv("VULNFEED_PATH"), "https://yadi-test.yandex-team.ru/db/{lang}.json.gz"),
			ManifestPath:    nonEmpty(os.Getenv("MANIFEST_PATH"), "https://yadi-test.yandex-team.ru/db/manifest.json"),
			RequestTimeout:  5,
			WatcherCronSpec: nonEmpty(os.Getenv("FEED_UPDATE_CRON_SPEC"), "0/10 * * * *"),
			WithDev:         true,
			HecSenderToken:  os.Getenv("YAUDIT_HEC_TOKEN"),
			WithCache:       true,
			Cacher: config.Cacher{
				Endpoint: "ydb-ru.yandex.net:2135",
				Database: "/ru/home/robot-yadi/mydb",
				Path:     "dev/yaudit",
			},
			Cleaner: config.Cleaner{
				YtProxy:     "locke",
				YtPath:      "//home/security/yaudit/dev",
				YtToken:     os.Getenv("YT_TOKEN"),
				StartDelay:  30 * time.Minute,
				CleanPeriod: 5 * time.Minute,
			},
		}

	case "prod":
		fallthrough

	default:
		simplelog.SetLevel(simplelog.WarnLevel)
		cfg = config.Config{
			Debug:           false,
			HTTPPort:        80,
			FeedPath:        nonEmpty(os.Getenv("VULNFEED_PATH"), "https://yadi.yandex-team.ru/db/{lang}.json.gz"),
			ManifestPath:    nonEmpty(os.Getenv("MANIFEST_PATH"), "https://yadi.yandex-team.ru/db/manifest.json"),
			RequestTimeout:  5,
			WatcherCronSpec: nonEmpty(os.Getenv("FEED_UPDATE_CRON_SPEC"), "0 * * * *"),
			WithDev:         false,
			HecSenderToken:  os.Getenv("YAUDIT_HEC_TOKEN"),
			WithCache:       true,
			Cacher: config.Cacher{
				Endpoint: "ydb-ru.yandex.net:2135",
				Database: "/ru/home/robot-yadi/mydb",
				Path:     "prod/yaudit",
			},
			Cleaner: config.Cleaner{
				YtProxy:     "locke",
				YtPath:      "//home/security/yaudit/prod",
				YtToken:     os.Getenv("YT_TOKEN"),
				StartDelay:  30 * time.Minute,
				CleanPeriod: 5 * time.Minute,
			},
		}
	}

	return cfg

}

func main() {
	_ = maxprocs.AdjustAuto()

	srv, err := server.New(newConfig())
	if err != nil {
		panic(err)
	}

	stopChan := make(chan os.Signal, 1)
	signal.Notify(stopChan, syscall.SIGINT, syscall.SIGTERM)

	errChan := make(chan error, 1)
	go func() {
		simplelog.Info("starting server")
		if err := srv.ListenAndServe(); err != nil {
			errChan <- err
		}
	}()

	select {
	case <-stopChan:
		ctx, cancel := context.WithTimeout(context.Background(), shutdownDeadline)
		defer cancel()

		if err := srv.Shutdown(ctx); err != nil {
			simplelog.Error("failed to shutdown NPM audit server", "err", err)
		}
	case err := <-errChan:
		simplelog.Error("unexpected server error", "err", err)
	}
}
