package baker

import (
	"fmt"
	"sync"

	"a.yandex-team.ru/security/yadi/libs/versionarium"
	"a.yandex-team.ru/security/yadi/yadi/pkg/feed"
)

type Bulkbaker struct {
	language string
	fetcher  *feed.Fetcher
	vulns    feed.Vulnerabilities
	lock     sync.RWMutex
}

func NewBacker(opts ...BakerOpt) (*Bulkbaker, error) {
	var err error
	bulkbaker := &Bulkbaker{}

	for _, opt := range opts {
		opt(bulkbaker)
	}

	bulkbaker.vulns, err = bulkbaker.fetcher.Fetch(bulkbaker.language)
	if err != nil {
		return nil, err
	}

	return bulkbaker, nil
}

func (b *Bulkbaker) ForPackage(pkg string) []feed.Vulnerability {
	b.lock.RLock()
	defer b.lock.RUnlock()

	return b.vulns.ForPackage(pkg)
}

func (b *Bulkbaker) ForPackageWithVersion(pkg string, version string) []feed.Vulnerability {
	b.lock.RLock()
	defer b.lock.RUnlock()

	parsedVersion, err := versionarium.NewVersion(b.language, version)
	if err != nil {
		return nil
	}

	pkgVulns := b.vulns.ForPackage(pkg)
	var vulns []feed.Vulnerability

	for _, vuln := range pkgVulns {
		if vuln.CheckVersion(parsedVersion) {
			vulns = append(vulns, vuln)
		}
	}

	return vulns
}

func (b *Bulkbaker) Reload() (err error) {
	b.lock.Lock()
	defer b.lock.Unlock()

	b.vulns, err = b.fetcher.Fetch(b.language)
	if err != nil {
		return fmt.Errorf("failed to reload feed")
	}
	return nil
}
