package cleaner

import (
	"context"
	"fmt"
	"time"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/manifestor"
	"a.yandex-team.ru/security/yadi/yaudit/internal/cacher"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
	"a.yandex-team.ru/yt/go/yterrors"
	"a.yandex-team.ru/yt/go/ytlock"
)

const (
	lockName = "clean-lock"
)

type (
	Source       = string
	Filename     = string
	SourcesFeeds map[Source]Filename
	Cleaner      struct {
		cacher       *cacher.Cacher
		manifestor   *manifestor.Manifestor
		yt           yt.Client
		ytPath       string
		sourcesFeeds SourcesFeeds
		startDelay   time.Duration
		cleanPeriod  time.Duration
	}
	Options struct {
		Manifestor   *manifestor.Manifestor
		YtProxy      string
		YtPath       string
		YtToken      string
		SourcesFeeds SourcesFeeds
		StartDelay   time.Duration
		CleanPeriod  time.Duration
	}
)

func NewCleaner(cacher *cacher.Cacher, opts Options) (*Cleaner, error) {
	c := &Cleaner{
		cacher:       cacher,
		sourcesFeeds: opts.SourcesFeeds,
		ytPath:       opts.YtPath,
		manifestor:   opts.Manifestor,
		startDelay:   opts.StartDelay,
		cleanPeriod:  opts.CleanPeriod,
	}

	var err error
	c.yt, err = ythttp.NewClient(&yt.Config{
		Proxy: opts.YtProxy,
		Token: opts.YtToken,
	})

	if err != nil {
		return nil, fmt.Errorf("can't create YT client: %w", err)
	}

	return c, nil
}

func (c *Cleaner) Start(ctx context.Context) {
	// TODO(buglloc): do smth better, for now we just "wait" all instances to up
	time.Sleep(c.startDelay)

	select {
	case <-ctx.Done():
		return
	case <-time.After(c.cleanPeriod):
		err := c.Cleanup(ctx)
		if err != nil {
			simplelog.Error("failed to cleanup cache", "err", err)
		}
	}
}

func (c *Cleaner) Cleanup(ctx context.Context) error {
	lock := ytlock.NewLockOptions(c.yt, ypath.Path(c.ytPath).JoinChild(lockName), ytlock.Options{
		CreateIfMissing: true,
		LockMode:        yt.LockExclusive,
	})

	_, err := lock.Acquire(ctx)
	if err != nil {
		if yterrors.ContainsErrorCode(err, yterrors.CodeConcurrentTransactionLockConflict) {
			return nil
		}
		return fmt.Errorf("failed to acquire lock: %w", err)
	}
	defer func() { _ = lock.Release(ctx) }()

	simplelog.Info("YT lock acquired, try to cleanup")

	cleanSourceEntries := func(ctx context.Context, source, epoch string) error {
		return c.cacher.CleanupEntries(ctx, source, epoch)
	}

	for source, filename := range c.sourcesFeeds {
		hash, ok := c.manifestor.Manifest.Hashes[filename]
		if !ok {
			continue
		}

		if err := cleanSourceEntries(ctx, source, hash); err != nil {
			simplelog.Error("failed to cleanup entries", "source", source, "epoch", hash, "err", err)
			continue
		}
		simplelog.Info("source cleaned", "source", source, "epoch", hash)
	}

	return nil
}
