package middlewares

import (
	"fmt"
	"regexp"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/versionarium"
)

const (
	language     = "nodejs"
	uaPrefix     = "npm"
	NpmFixedVer  = "6.7.0" // https://github.com/npm/npm-audit-report/pull/34
	NpmFixedFlag = "NPM-FIXED-HEADER"
	SemVerRegex  = `v?([0-9]+)(\.[0-9]+)?(\.[0-9]+)?` +
		`(-([0-9A-Za-z\-]+(\.[0-9A-Za-z\-]+)*))?` +
		`(\+([0-9A-Za-z\-]+(\.[0-9A-Za-z\-]+)*))?`
)

var NpmUserAgentRE *regexp.Regexp
var NpmVerConstraint versionarium.VersionRange

func init() {
	NpmUserAgentRE, _ = regexp.Compile(fmt.Sprintf("%s/(%s)", uaPrefix, SemVerRegex))
	NpmVerConstraint, _ = versionarium.NewRange(language, fmt.Sprintf(">=%s", NpmFixedVer))
}

func CheckNpmVersion() echo.MiddlewareFunc {
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) error {

			var result bool
			req := c.Request()

			userAgent := req.UserAgent()
			npmVersion := NpmUserAgentRE.FindStringSubmatch(userAgent)

			if len(npmVersion) < 1 {
				result = false
				simplelog.Error(fmt.Sprintf("Can not parse npm version from User-Agent: %s", userAgent))
			} else {
				currentVer, err := versionarium.NewVersion(language, npmVersion[1])
				if err != nil {
					result = false
					simplelog.Error(fmt.Sprintf("Can not create npm semver from User-Agent: %s", userAgent))
				} else {
					result = NpmVerConstraint.Check(currentVer)
				}
			}

			c.Set(NpmFixedFlag, result)
			return next(c)
		}
	}
}

func Fixed(ctx echo.Context) bool {
	result := false
	flag := ctx.Get(NpmFixedFlag)
	if flag.(bool) {
		result = true
	}
	return result
}
