package npmaudit

import (
	"context"
	"sync"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/yadi/pkg/analyze"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/npm"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/npmaudit"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/pkglock"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/yarnaudit"
)

type (
	AnalyzerOpts struct {
		PkgJSON  *npm.PackageJSON
		PkgLock  *pkglock.LockJSON
		YarnLock []byte
		Ctx      context.Context
		WithDev  bool
	}

	Analyzer struct {
		yadi      analyze.Analyzer
		savedOpts []analyze.AnalyzerOption
		lock      sync.RWMutex
	}
)

func NewAnalyzer(opts ...analyze.AnalyzerOption) (*Analyzer, error) {
	return &Analyzer{
		yadi:      analyze.NewAnalyzer(opts...),
		savedOpts: opts,
	}, nil
}

func (a *Analyzer) AnalyzePkgLock(opts AnalyzerOpts) (analyze.ResultAnalyze, error) {
	a.lock.RLock()
	defer a.lock.RUnlock()

	manager, err := npmaudit.NewManager(npmaudit.ManagerOpts{
		PkgJSON:     opts.PkgJSON,
		PkgLockJSON: opts.PkgLock,
		WithDev:     opts.WithDev,
	})

	if err != nil {
		simplelog.Error("Failed to create new manager", "err", err)
		return analyze.ResultAnalyze{}, err
	}

	return a.yadi.Analyze(opts.Ctx, analyze.Request{
		PackageManager: manager,
	})
}

func (a *Analyzer) AnalyzeYarnLock(opts AnalyzerOpts) (analyze.ResultAnalyze, error) {
	a.lock.RLock()
	defer a.lock.RUnlock()

	manager, err := yarnaudit.NewManager(yarnaudit.ManagerOpts{
		PkgJSON:  opts.PkgJSON,
		YarnLock: opts.YarnLock,
		WithDev:  opts.WithDev,
	})

	if err != nil {
		simplelog.Error("Failed to create new manager", "err", err)
		return analyze.ResultAnalyze{}, err
	}

	return a.yadi.Analyze(opts.Ctx, analyze.Request{
		PackageManager: manager,
	})
}

func (a *Analyzer) AnalyzePkg(ctx context.Context, pkgName, pkgVer string) (analyze.ResultAnalyzePkg, error) {
	a.lock.RLock()
	defer a.lock.RUnlock()

	return a.yadi.AnalyzePkg(ctx, analyze.PkgRequest{
		PackageManager: new(npmaudit.Manager),
		PackageName:    pkgName,
		PackageVersion: pkgVer,
	})
}

func (a *Analyzer) Reload() error {
	a.lock.Lock()
	defer a.lock.Unlock()

	a.yadi = analyze.NewAnalyzer(a.savedOpts...)
	return nil
}
