package npmaudit

import (
	"errors"
	"fmt"
	"net/url"
	"path"
	"regexp"
	"strings"

	"a.yandex-team.ru/security/yadi/libs/cvs"
	"a.yandex-team.ru/security/yadi/yadi/pkg/analyze"
)

const (
	YadiWebVulnURL = "https://v.ya.cc/"
)

var (
	SemverRE = regexp.MustCompile(`\@(\d+\.){2}(\d+)`)
)

type (
	IssueResolving struct {
		ID   string `json:"id"`
		Path string `json:"path"`
	}

	ReportActions struct {
		Action   string           `json:"action"`
		Module   string           `json:"module"`
		Target   string           `json:"target"`
		Resolves []IssueResolving `json:"resolves"`
	}

	IssPath struct {
		Version  string   `json:"version"`
		Paths    []string `json:"paths"`
		Dev      bool     `json:"dev"`
		Optional bool     `json:"optional"`
		Bundled  bool     `json:"bundled"`
	}

	Issue struct {
		ID                 string    `json:"id"`
		IssueName          string    `json:"title"`
		PkgName            string    `json:"module_name"`
		VulnerableVersions string    `json:"vulnerable_versions"`
		PatchedVersions    string    `json:"patched_versions"`
		Severity           string    `json:"severity"`
		Findings           []IssPath `json:"findings"`
		URL                string    `json:"url"`
	}

	VulnsCount struct {
		Info     int `json:"info"`
		Low      int `json:"low"`
		Moderate int `json:"moderate"`
		High     int `json:"high"`
		Critical int `json:"critical"`
	}

	ReportMetadata struct {
		Vulns     VulnsCount `json:"vulnerabilities"`
		Deps      uint64     `json:"dependencies"`
		DevDeps   uint64     `json:"devDependencies"`
		OptDeps   uint64     `json:"optionalDependencies"`
		TotalDeps uint64     `json:"totalDependencies"`
	}

	Report struct {
		Actions    []*ReportActions  `json:"actions"`
		Advisories map[string]*Issue `json:"advisories"`
		Meta       ReportMetadata    `json:"metadata"`
	}
)

func NewReport() *Report {
	return &Report{
		Advisories: map[string]*Issue{},
		Actions:    []*ReportActions{},
		Meta:       ReportMetadata{},
	}
}

func (r *Report) Generate(yadiResult analyze.ResultAnalyze, quick bool, npmfixed bool) (err error) {

	r.Advisories = map[string]*Issue{}
	r.Actions = []*ReportActions{}
	r.Meta = ReportMetadata{}

	// check YADI issues
	var issues analyze.IssueList
	if len(yadiResult.Issues) > 0 {
		for _, iss := range yadiResult.Issues {
			issues = iss
			break // because there is only one project
		}
	} else {
		return errors.New("empty YADI issues")
	}

	// check YADI stats
	var stats analyze.DependencyStats
	if len(yadiResult.Stats) > 0 {
		for _, st := range yadiResult.Stats {
			stats = st
			break // because there is only one project
		}
	} else {
		return errors.New("empty YADI stats")
	}

	// generate meta
	r.Meta.TotalDeps, r.Meta.DevDeps = stats.Stats()
	r.Meta.Deps = r.Meta.TotalDeps - r.Meta.DevDeps

	if len(issues) == 0 {
		return nil
	}

	// generate advisories
	for _, issue := range issues {
		if _, exists := r.Advisories[issue.Vulnerability.ID]; !exists {

			var title string
			vulnURL, _ := url.Parse(YadiWebVulnURL)
			vulnURL.Path = path.Join(vulnURL.Path, issue.Vulnerability.ID)
			if npmfixed {
				title = fmt.Sprintf("%s.", issue.Summary)
			} else {
				title = fmt.Sprintf("%s. More info: %s", issue.Summary, vulnURL.String())
			}

			patchedVersions := issue.Vulnerability.PatchedVersions
			if patchedVersions == "" {
				patchedVersions = "<0.0.0" // https://github.com/npm/npm-audit-report/blob/latest/reporters/detail.js#L155
			}
			r.Advisories[issue.Vulnerability.ID] = &Issue{
				ID:                 issue.Vulnerability.ID,
				IssueName:          title,
				PkgName:            issue.PackageName,
				VulnerableVersions: issue.Vulnerability.RawVersions,
				PatchedVersions:    patchedVersions,
				Severity:           NpmSeverity(issue.CVSSScore),
				URL:                vulnURL.String(),
				Findings: append([]IssPath{}, IssPath{
					Version: issue.Version,
					Paths:   []string{semverToPath(issue.Path)},
				}),
			}

			// low|high|moderate|critical ??
			switch issue.Severity() {
			case "Low":
				r.Meta.Vulns.Low++
			case "Medium":
				r.Meta.Vulns.Moderate++ // ???
			case "High":
				r.Meta.Vulns.High++
			case "Critical":
				r.Meta.Vulns.Critical++
			}

		} else {
			for i, f := range r.Advisories[issue.Vulnerability.ID].Findings {
				if f.Version == issue.Version {
					r.Advisories[issue.Vulnerability.ID].Findings[i].Paths = append(
						r.Advisories[issue.Vulnerability.ID].Findings[i].Paths,
						semverToPath(issue.Path),
					)
					break
				} else {
					r.Advisories[issue.Vulnerability.ID].Findings = append(r.Advisories[issue.Vulnerability.ID].Findings, IssPath{
						Version: issue.Version,
						Paths:   []string{semverToPath(issue.Path)},
					})
				}
			}
		}
	}

	// generate actions
	if !quick {
		for _, iss := range r.Advisories {
			var resolves []IssueResolving
			for _, f := range iss.Findings {
				for _, p := range f.Paths {
					resolves = append(resolves, IssueResolving{
						ID:   iss.ID,
						Path: p,
					})
				}
			}

			r.Actions = append(r.Actions, &ReportActions{
				Action: "review", // TODO(melkikh): not only manual review
				Module: iss.PkgName,
				// TODO: implement it!
				//Target:   iss.PatchedVersions,
				Resolves: resolves,
			})
		}
	}

	return nil

}

func semverToPath(path []string) string {
	return SemverRE.ReplaceAllLiteralString(strings.Join(path, string('\x3e')), "")
}

func NpmSeverity(score float32) string {
	switch cvs.RoundBySeverity(score) {
	case cvs.LowSeverity:
		return "low"
	case cvs.MediumSeverity:
		return "moderate"
	case cvs.HighSeverity:
		return "high"
	case cvs.CriticalSeverity:
		return "critical"
	default:
		return "low"
	}
}
