package pyaudit

import (
	"context"
	"regexp"
	"sync"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/yadi/yadi/pkg/analyze"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/pip"
)

type (
	AnalyzerOpts struct {
		Ctx        context.Context
		PkgName    string
		PkgVersion string
	}

	Analyzer struct {
		yadi      analyze.Analyzer
		savedOpts []analyze.AnalyzerOption
		lock      sync.RWMutex
	}
)

var (
	ErrPkgNotFound = analyze.ErrPkgNotFound
	clearVersionRe = regexp.MustCompile(`^v?((?:[0-9xX*]+)(?:\.[0-9xX*]+)?(?:\.[0-9xX*]+)?)[^,;:#]*`)
)

func NewAnalyzer(opts ...analyze.AnalyzerOption) (*Analyzer, error) {
	return &Analyzer{
		yadi:      analyze.NewAnalyzer(opts...),
		savedOpts: opts,
	}, nil
}

func (a *Analyzer) Analyze(opts AnalyzerOpts) (analyze.ResultAnalyzePkg, error) {
	a.lock.RLock()
	defer a.lock.RUnlock()

	pipManager, err := pip.NewManager(pip.ManagerOpts{
		WithDev:          false,
		ResolveMode:      manager.ResolveRemote,
		WithoutPypiCache: false,
	})

	if err != nil {
		return analyze.ResultAnalyzePkg{}, nil
	}

	clearedVersion := clearVersionRe.FindStringSubmatch(opts.PkgVersion)
	if len(clearedVersion) == 0 {
		return analyze.ResultAnalyzePkg{}, xerrors.Errorf("failed to parse version")
	}

	return a.yadi.AnalyzePkg(opts.Ctx, analyze.PkgRequest{
		PackageManager: pipManager,
		PackageName:    opts.PkgName,
		PackageVersion: clearedVersion[1],
	})
}

func (a *Analyzer) List(opts AnalyzerOpts) (analyze.ResultListPkg, error) {
	a.lock.RLock()
	defer a.lock.RUnlock()

	pipManager, err := pip.NewManager(pip.ManagerOpts{
		WithDev:          false,
		ResolveMode:      manager.ResolveRemote,
		WithoutPypiCache: false,
	})

	if err != nil {
		return analyze.ResultListPkg{}, nil
	}

	clearedVersion := clearVersionRe.FindStringSubmatch(opts.PkgVersion)
	if len(clearedVersion) == 0 {
		return analyze.ResultListPkg{}, xerrors.Errorf("failed to parse version")
	}

	return a.yadi.ListPkg(opts.Ctx, analyze.ListPkgRequest{
		PackageManager: pipManager,
		PackageName:    opts.PkgName,
		PackageVersion: clearedVersion[1],
	})
}

func (a *Analyzer) Reload() error {
	a.lock.Lock()
	defer a.lock.Unlock()

	a.yadi = analyze.NewAnalyzer(a.savedOpts...)
	return nil
}
