package server

import (
	"fmt"
	"net/http"

	"github.com/labstack/echo/v4"
	"github.com/mailru/easyjson"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/yadi/pkg/analyze"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/npm"
	"a.yandex-team.ru/security/yadi/yadi/pkg/manager/pkglock"
	"a.yandex-team.ru/security/yadi/yaudit/internal/middlewares"
	"a.yandex-team.ru/security/yadi/yaudit/internal/npmaudit"
)

type (
	npmMultipleCheckReq struct {
		Versions []string `json:"versions"`
	}

	npmMultipleCheckRsp struct {
		Issues map[string]analyze.ResultAnalyzePkg `json:"issues"`
	}

	npmAdvisoriesBulkReq map[string][]string
	npmAdvisoriesBulkRsp map[string][]npmaudit.Advisory
)

func (s *Server) npmAdvisoriesBulkHandler(c echo.Context) error {
	s.unistat.AddRequest()
	s.unistat.AddAdvisoriesBulk()

	var req npmAdvisoriesBulkReq

	if err := c.Bind(&req); err != nil {
		simplelog.Error("Failed to parse bulk JSON", "err", err)
		return echo.NewHTTPError(http.StatusInternalServerError, "could not parse JSON")
	}

	rsp := make(npmAdvisoriesBulkRsp)

	for pkg, versions := range req {
		for _, version := range versions {
			vulns := s.baker.ForPackageWithVersion(pkg, version)
			if len(vulns) == 0 {
				continue
			}

			advisories := make([]npmaudit.Advisory, len(vulns))
			for i, vuln := range vulns {
				advisories[i] = npmaudit.Advisory{
					ID:                 vuln.ID,
					IssueName:          vuln.Summary,
					VulnerableVersions: vuln.RawVersions,
					Severity:           npmaudit.NpmSeverity(vuln.CVSSScore),
					URL:                vuln.Reference,
				}
			}
			rsp[pkg] = advisories
		}
	}

	if len(rsp) > 0 {
		s.unistat.AddVulnerable()
	} else {
		s.unistat.AddNotVulnerable()
	}

	return apiOk(c, rsp)
}

func (s *Server) npmAuditHandler(quick bool) echo.HandlerFunc {
	return func(c echo.Context) (err error) {
		s.unistat.AddRequest()

		if quick {
			s.unistat.AddInstall()
		} else {
			s.unistat.AddAudit()
		}

		var req npmaudit.NpmRequest
		if err := easyjson.UnmarshalFromReader(c.Request().Body, &req); err != nil {
			simplelog.Error("Failed to parse audit JSON", "err", err)
			return echo.NewHTTPError(http.StatusInternalServerError, "could not parse JSON")
		}

		if req.Dependencies == nil || len(req.Requires) == 0 {
			simplelog.Info("Project with empty dependencies")
			empty := npmaudit.NewReport()
			return c.JSON(http.StatusOK, empty)
		}

		yadiResults, err := s.npmAnalyzer.AnalyzePkgLock(npmaudit.AnalyzerOpts{
			Ctx:     c.Request().Context(),
			WithDev: s.cfg.WithDev,
			PkgJSON: &npm.PackageJSON{
				Name:         req.AppName,
				RawVersion:   req.RawVersion,
				Dependencies: req.Requires,
			},
			PkgLock: &pkglock.LockJSON{
				Name:            req.AppName,
				Version:         req.RawVersion,
				LockfileVersion: pkglock.LockVersion,
				Dependencies:    *req.Dependencies,
			},
		})

		if err != nil {
			simplelog.Error("Failed to analyze project", "err", err)
			if err == analyze.ErrCanceled {
				s.unistat.AddTimeout()
			}
			return echo.ErrInternalServerError
		}

		report := npmaudit.NewReport()

		err = report.Generate(yadiResults, quick, middlewares.Fixed(c))
		if err != nil {
			simplelog.Error("Failed to generate report", "err", err)
			return echo.ErrInternalServerError
		}

		if len(report.Advisories) > 0 {
			s.unistat.AddVulnerable()
		} else {
			s.unistat.AddNotVulnerable()
		}

		return apiOk(c, report)
	}
}

func (s *Server) npmCheckHandler(c echo.Context) error {
	pkgName := c.Param("name")
	pkgVersion := c.Param("version")
	if pkgVersion == "" {
		pkgVersion = "0.0.0"
	}

	result, err := s.npmAnalyzer.AnalyzePkg(c.Request().Context(), pkgName, pkgVersion)
	if err != nil {
		return apiErr(c, err)
	}

	return apiOk(c, result)
}

func (s *Server) npmMultipleCheckHandler(c echo.Context) error {
	var req npmMultipleCheckReq
	if err := c.Bind(&req); err != nil {
		return echo.NewHTTPError(http.StatusInternalServerError, fmt.Sprintf("could not parse JSON: %s", err))
	}

	pkgName := c.Param("name")
	rsp := npmMultipleCheckRsp{
		Issues: make(map[string]analyze.ResultAnalyzePkg, len(req.Versions)),
	}

	for _, version := range req.Versions {
		result, err := s.npmAnalyzer.AnalyzePkg(c.Request().Context(), pkgName, version)
		if err != nil {
			return apiErr(c, err)
		}
		rsp.Issues[version] = result
	}

	return apiOk(c, rsp)
}
