package server

import (
	"context"
	"encoding/hex"
	"io/ioutil"
	"net/http"

	"github.com/labstack/echo/v4"
	"github.com/mailru/easyjson"
	"github.com/mailru/easyjson/jwriter"
	"golang.org/x/crypto/blake2s"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/yadi/pkg/analyze"
	"a.yandex-team.ru/security/yadi/yaudit/internal/cacher"
	"a.yandex-team.ru/security/yadi/yaudit/internal/npmaudit"
)

const (
	okoSource = "oko"
)

func (s *Server) okoHandler(c echo.Context) (err error) {
	body, err := ioutil.ReadAll(c.Request().Body)
	if err != nil {
		return echo.NewHTTPError(http.StatusInternalServerError, err.Error())
	}

	ctx := c.Request().Context()
	reqHash := blakeHash(body)
	if s.cacher != nil {
		if res, err := s.cacher.LookupEntry(ctx, okoSource, reqHash); err == nil {
			c.Response().Header().Set("X-Cached", "yes")
			return c.Blob(http.StatusOK, echo.MIMEApplicationJSONCharsetUTF8, res)
		}
	}

	okoResult, err := s.doOko(ctx, body)
	if err != nil {
		simplelog.Error("failed to process OKO request", "err", err)
		return echo.NewHTTPError(http.StatusInternalServerError, err.Error())
	}

	w := jwriter.Writer{
		Flags: jwriter.NilSliceAsEmpty,
	}
	okoResult.MarshalEasyJSON(&w)
	results, err := w.BuildBytes()
	if err != nil {
		simplelog.Error("failed to marshal OKO result", "err", err)
		return echo.NewHTTPError(http.StatusInternalServerError, err.Error())
	}

	if s.cacher != nil {
		err := s.cacher.UpsertEntry(ctx, cacher.Entry{
			Source:  okoSource,
			Hash:    reqHash,
			Epoch:   s.manifestor.Manifest.Hashes[SourcesFeeds[okoSource]],
			Results: results,
		})
		if err != nil {
			simplelog.Error("failed to save OKO result", "err", err)
		}
	}

	return c.Blob(http.StatusOK, echo.MIMEApplicationJSONCharsetUTF8, results)
}

func (s *Server) doOko(ctx context.Context, body []byte) (analyze.IssueList, error) {
	var req npmaudit.OkoRequest
	err := easyjson.Unmarshal(body, &req)
	if err != nil {
		return nil, err
	}

	var analyzerResults analyze.ResultAnalyze
	if len(req.YarnLock) != 0 {
		analyzerResults, err = s.npmAnalyzer.AnalyzeYarnLock(npmaudit.AnalyzerOpts{
			Ctx:      ctx,
			WithDev:  req.WithDev,
			PkgJSON:  &req.PackageJSON,
			YarnLock: []byte(req.YarnLock),
		})
	} else {
		analyzerResults, err = s.npmAnalyzer.AnalyzePkgLock(npmaudit.AnalyzerOpts{
			Ctx:     ctx,
			WithDev: req.WithDev,
			PkgJSON: &req.PackageJSON,
			PkgLock: &req.PackageLockJSON,
		})
	}

	if err != nil {
		return nil, err
	}

	var result analyze.IssueList
	for _, iss := range analyzerResults.Issues {
		result = iss
		break // because there is only one project
	}

	return result, nil
}

func blakeHash(in []byte) string {
	sum := blake2s.Sum256(in)
	return hex.EncodeToString(sum[:])
}
