package server

import (
	"context"
	"net/http"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/yaudit/internal/pyaudit"
)

func (s *Server) pythonCheckHandler(c echo.Context) error {
	pkgName := c.Param("name")
	pkgVersion := c.Param("version")
	if pkgVersion == "" {
		pkgVersion = "*"
	}

	ctx, cancel := context.WithTimeout(c.Request().Context(), time.Duration(s.cfg.RequestTimeout)*time.Second)
	defer cancel()

	result, err := s.pyAnalyzer.Analyze(pyaudit.AnalyzerOpts{
		Ctx:        ctx,
		PkgName:    pkgName,
		PkgVersion: pkgVersion,
	})

	if xerrors.Is(err, pyaudit.ErrPkgNotFound) {
		return apiNotFound(c, err)
	}

	if err != nil {
		return apiErr(c, err)
	}

	return apiOk(c, result)
}

func (s *Server) pythonListHandler(c echo.Context) error {
	pkgName := c.Param("name")
	pkgVersion := c.Param("version")
	if pkgVersion == "" {
		pkgVersion = "*"
	}

	ctx, cancel := context.WithTimeout(c.Request().Context(), time.Duration(s.cfg.RequestTimeout)*time.Second)
	defer cancel()

	result, err := s.pyAnalyzer.List(pyaudit.AnalyzerOpts{
		Ctx:        ctx,
		PkgName:    pkgName,
		PkgVersion: pkgVersion,
	})

	if xerrors.Is(err, pyaudit.ErrPkgNotFound) {
		return apiNotFound(c, err)
	}

	if err != nil {
		return apiErr(c, err)
	}

	return apiOk(c, result)
}

func apiOk(c echo.Context, result interface{}) error {
	return c.JSON(http.StatusOK, result)
}

func apiErr(c echo.Context, err error) error {
	simplelog.Error("failed to process API request", "url", c.Request().URL, "err", err)

	return c.JSON(http.StatusInternalServerError, echo.Map{
		"error": err.Error(),
	})
}

func apiNotFound(c echo.Context, err error) error {
	return c.JSON(http.StatusNotFound, echo.Map{
		"error": err.Error(),
	})
}
