package server

import (
	"context"
	"fmt"

	"github.com/labstack/echo/v4"
	"github.com/labstack/echo/v4/middleware"

	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/security/yadi/libs/cvs"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/manifestor"
	osanalyze "a.yandex-team.ru/security/yadi/yadi-os/pkg/analyze"
	osfeed "a.yandex-team.ru/security/yadi/yadi-os/pkg/feed"
	"a.yandex-team.ru/security/yadi/yadi/pkg/analyze"
	"a.yandex-team.ru/security/yadi/yadi/pkg/feed"
	"a.yandex-team.ru/security/yadi/yaudit/internal/baker"
	"a.yandex-team.ru/security/yadi/yaudit/internal/cacher"
	"a.yandex-team.ru/security/yadi/yaudit/internal/cleaner"
	"a.yandex-team.ru/security/yadi/yaudit/internal/config"
	"a.yandex-team.ru/security/yadi/yaudit/internal/middlewares"
	"a.yandex-team.ru/security/yadi/yaudit/internal/npmaudit"
	"a.yandex-team.ru/security/yadi/yaudit/internal/pyaudit"
	"a.yandex-team.ru/security/yadi/yaudit/internal/ubuntuaudit"
	"a.yandex-team.ru/security/yadi/yaudit/internal/unistat"
	"a.yandex-team.ru/security/yadi/yaudit/pkg/watcher"
)

type (
	Server struct {
		ctx            context.Context
		cancelCtx      context.CancelFunc
		cfg            config.Config
		echo           *echo.Echo
		unistat        *unistat.Counter
		tvmtool        *tvmtool.Client
		cacher         *cacher.Cacher
		cleaner        *cleaner.Cleaner
		baker          *baker.Bulkbaker
		watcher        *watcher.FeedWatcher
		npmAnalyzer    *npmaudit.Analyzer
		pyAnalyzer     *pyaudit.Analyzer
		ubuntuAnalyzer *ubuntuaudit.Analyzer
		manifestor     *manifestor.Manifestor
	}
)

var SourcesFeeds = cleaner.SourcesFeeds{
	okoSource: "nodejs.json.gz",
}

func New(cfg config.Config) (*Server, error) {

	counter := unistat.NewCounter()

	manifestManager := manifestor.New(cfg.ManifestPath)

	feedFetcher := feed.New(feed.Options{
		MinimumSeverity: cvs.MediumSeverity,
		FeedURI:         cfg.FeedPath,
	})

	osFeedFetcher := osfeed.NewFetcher(osfeed.Options{
		MinimumSeverity: cvs.MediumSeverity,
		FeedURI:         cfg.FeedPath,
	})

	bulkbaker, err := baker.NewBacker(
		baker.WithFetcher(feedFetcher),
		baker.WithLanguage("nodejs"),
	)
	if err != nil {
		return nil, fmt.Errorf("failed to init bulker: %w", err)
	}

	npmAnalyzer, err := npmaudit.NewAnalyzer(
		analyze.WithFeedFetcher(feedFetcher),
		analyze.WithSuggest(false),
		analyze.WithStatsTracking(true),
	)
	if err != nil {
		return nil, fmt.Errorf("failed to init NPM analyzer: %w", err)
	}

	pyAnalyzer, err := pyaudit.NewAnalyzer(
		analyze.WithFeedOptions(feed.Options{
			MinimumSeverity: cvs.MediumSeverity,
			FeedURI:         cfg.FeedPath,
		}),
		analyze.WithSuggest(false),
		analyze.WithStatsTracking(true),
	)
	if err != nil {
		return nil, fmt.Errorf("failed to init Python analyzer: %w", err)
	}

	ubuntuAnalyzer, err := ubuntuaudit.NewAnalyzer(osanalyze.Options{
		Fetcher:     osFeedFetcher,
		FixableOnly: true,
	})
	if err != nil {
		return nil, fmt.Errorf("failed to init Ubuntu analyzer: %w", err)
	}

	feedWatcher, err := watcher.NewWatcher(
		watcher.WithManifestManager(manifestManager),
		watcher.WithFeedToWatch(cfg.FeedPath),
		watcher.WithCronSchedule(cfg.WatcherCronSpec),
		watcher.WithFetchers(feedFetcher, osFeedFetcher),
		watcher.WithFailCounter(counter.AddUpdateFail),
		watcher.WithReloadHandler("nodejs", "bulker", bulkbaker.Reload),
		watcher.WithReloadHandler("nodejs", "npmaudit", npmAnalyzer.Reload),
		watcher.WithReloadHandler("python", "pyaudit", pyAnalyzer.Reload),
		watcher.WithReloadHandler("linux-ubuntu", "ubuntuaudit", ubuntuAnalyzer.Reload),
	)
	if err != nil {
		return nil, fmt.Errorf("failed to init watcher: %w", err)
	}

	e := echo.New()
	e.HideBanner = true
	e.Debug = cfg.Debug

	ctx, cancelFn := context.WithCancel(context.Background())
	srv := &Server{
		ctx:            ctx,
		cancelCtx:      cancelFn,
		cfg:            cfg,
		echo:           e,
		unistat:        counter,
		baker:          bulkbaker,
		watcher:        feedWatcher,
		npmAnalyzer:    npmAnalyzer,
		pyAnalyzer:     pyAnalyzer,
		ubuntuAnalyzer: ubuntuAnalyzer,
		manifestor:     manifestManager,
	}

	// init manifest
	if _, err = manifestManager.Update(ctx); err != nil {
		return nil, fmt.Errorf("failed to init start manifest: %w", err)
	}

	if !cfg.WithCache {
		return srv, nil
	}

	srv.tvmtool, err = tvmtool.NewAnyClient()
	if err != nil {
		return nil, err
	}

	srv.cacher, err = cacher.New(srv.ctx, srv.tvmtool, cacher.Options(cfg.Cacher))
	if err != nil {
		return nil, err
	}

	srv.cleaner, err = cleaner.NewCleaner(srv.cacher, cleaner.Options{
		YtProxy:      cfg.Cleaner.YtProxy,
		YtPath:       cfg.Cleaner.YtPath,
		YtToken:      cfg.Cleaner.YtToken,
		StartDelay:   cfg.Cleaner.StartDelay,
		CleanPeriod:  cfg.Cleaner.CleanPeriod,
		Manifestor:   manifestManager,
		SourcesFeeds: SourcesFeeds,
	})
	if err != nil {
		return nil, err
	}

	return srv, nil
}

func (s *Server) ListenAndServe() error {
	if s.cleaner != nil {
		go s.cleaner.Start(s.ctx)
	}

	if err := s.watcher.Watch(s.ctx); err != nil {
		return fmt.Errorf("failed to start watcher: %w", err)
	}

	s.echo.GET("/ping", s.pingHandler)
	s.echo.GET("/version", s.versionHandler)
	s.echo.GET("/unistat", s.unistatHandler)

	// NPM Audit API routes
	// https://docs.npmjs.com/cli/v7/commands/npm-audit#audit-endpoints
	npm := s.echo.Group("/-/npm/v1/security")
	{
		npm.Use(middlewares.Gunzip())
		npm.Use(middleware.Gzip())
		npm.Use(middlewares.CheckNpmVersion())
		npm.Use(middleware.RequestID())

		npm.POST("/audits", s.npmAuditHandler(false))
		npm.POST("/audits/quick", s.npmAuditHandler(true))
		npm.POST("/advisories/bulk", s.npmAdvisoriesBulkHandler)
	}

	apiGroup := s.echo.Group("/api/v1")
	{
		python := apiGroup.Group("/nodejs")
		python.POST("/check", s.okoHandler)
		python.GET("/check/:name", s.npmCheckHandler)
		python.POST("/check/:name", s.npmMultipleCheckHandler)
		python.GET("/check/:name/:version", s.npmCheckHandler)
	}

	{
		python := apiGroup.Group("/python")
		python.GET("/check/:name", s.pythonCheckHandler)
		python.GET("/check/:name/:version", s.pythonCheckHandler)
		python.GET("/list/:name", s.pythonListHandler)
		python.GET("/list/:name/:version", s.pythonListHandler)
	}

	{
		ubuntu := apiGroup.Group("/ubuntu")
		ubuntu.Use(middlewares.Gunzip())
		ubuntu.Use(middleware.Gzip())

		ubuntu.GET("/check/:name", s.ubuntuCheckHandler)
		ubuntu.GET("/check/:name/:version", s.ubuntuCheckHandler)
		ubuntu.POST("/dpkg", s.ubuntuDPKGHandler)
	}

	return s.echo.Start(fmt.Sprintf(":%d", s.cfg.HTTPPort))
}

func (s *Server) Shutdown(ctx context.Context) error {
	defer func() {
		if s.cacher != nil {
			_ = s.cacher.Close(s.ctx)
		}
		s.watcher.Stop()
		s.cancelCtx()
	}()

	return s.echo.Shutdown(ctx)
}
