package watcher

import (
	"context"
	"crypto/tls"
	"fmt"
	"strings"
	"time"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/drive/library/go/cron"
	"a.yandex-team.ru/library/go/certifi"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/manifestor"
	"a.yandex-team.ru/security/yadi/yadi/pkg/feed"
)

const defaultFeedURI = "https://yadi.yandex-team.ru/db/{lang}.json"

type (
	ReloadFunc     = func() error
	reloadFuncSpec struct {
		name, language, lastHash string
		f                        ReloadFunc
	}

	FeedWatcher struct {
		feedURI            string
		nextTime           time.Time
		cancelFunc         context.CancelFunc
		reloadHandlerSpecs []reloadFuncSpec
		httpc              *resty.Client
		manifestor         *manifestor.Manifestor
		scheduler          *cron.Schedule
		timer              *time.Timer
		fetchers           []FetchCleaner
		fail               func()
	}

	FetchCleaner interface {
		CleanCache(language string)
	}
)

func NewWatcher(opts ...Option) (*FeedWatcher, error) {
	hourlyScheduler, _ := cron.Parse(hourly)

	watcher := &FeedWatcher{
		manifestor:         manifestor.New(""),
		feedURI:            defaultFeedURI,
		reloadHandlerSpecs: []reloadFuncSpec{},
		scheduler:          hourlyScheduler,
		fail:               func() {},
	}

	for _, opt := range opts {
		opt(watcher)
	}

	certPool, err := certifi.NewCertPool()
	if err != nil {
		return nil, fmt.Errorf("can't create ca pool: %w", err)
	}

	watcher.httpc = resty.New().
		SetTLSClientConfig(&tls.Config{RootCAs: certPool}).
		SetBaseURL(defaultFeedURI).
		SetRetryCount(5).
		SetRetryWaitTime(1 * time.Second).
		SetRetryMaxWaitTime(100 * time.Second).
		AddRetryCondition(func(rsp *resty.Response, err error) bool {
			return err != nil
		})

	return watcher, nil
}

func (w *FeedWatcher) Watch(ctx context.Context) error {
	// updating manifest at start
	manifest, err := w.manifestor.Update(ctx)
	if err != nil {
		return fmt.Errorf("failed to init manifest: %w", err)
	}

	// init correct "lastHashes"
	for i := range w.reloadHandlerSpecs {
		filename := parseFeedFilename(w.feedURI, w.reloadHandlerSpecs[i].language)
		if hashFromManifest, ok := manifest.Hashes[filename]; ok {
			w.reloadHandlerSpecs[i].lastHash = hashFromManifest
		}
	}

	start := time.Now()
	w.nextTime = w.scheduler.Next(start)
	w.timer = time.NewTimer(time.Until(w.nextTime))
	go w.loop(ctx)
	return nil
}

func (w *FeedWatcher) loop(ctx context.Context) {
	for {
		select {
		case <-w.timer.C:
			now := time.Now()
			w.nextTime = w.scheduler.Next(now)
			w.timer.Reset(time.Until(w.nextTime))
			simplelog.Info("start updating feeds", "now", now.String(), "next", w.nextTime.String())
			w.feedUpdateWorker(ctx)
		case <-ctx.Done():
			return
		}
	}
}

func (w *FeedWatcher) Stop() {
	w.timer.Stop()
}

func (w *FeedWatcher) feedUpdateWorker(ctx context.Context) {
	manifest, err := w.manifestor.Update(ctx)
	if err != nil {
		simplelog.Error("failed to load manifest", "err", err)
		w.fail()
		return
	}

	var cleanedCaches = make(map[string]struct{})
	updateFeed := func(i int) error {
		spec := w.reloadHandlerSpecs[i]
		fileName := parseFeedFilename(w.feedURI, spec.language)
		newHash, ok := manifest.Hashes[fileName]
		if !ok {
			return fmt.Errorf("feed %s hash wasn't found in manifest", fileName)
		}

		if manifest.Hashes[fileName] == spec.lastHash {
			return nil
		}

		if _, cleaned := cleanedCaches[spec.language]; !cleaned {
			for _, f := range w.fetchers {
				f.CleanCache(spec.language)
			}

			cleanedCaches[spec.language] = struct{}{}
		}

		if err := spec.f(); err != nil {
			return fmt.Errorf("failed to reload %s feed: %w", spec.name, err)
		}
		w.reloadHandlerSpecs[i].lastHash = newHash
		return nil
	}

	for i := range w.reloadHandlerSpecs {
		simplelog.Info("start updating feed", "consumer", w.reloadHandlerSpecs[i].name, "hash", w.reloadHandlerSpecs[i].lastHash)
		err := updateFeed(i)
		if err != nil {
			simplelog.Error("failed to update feed", "err", err)
			w.fail()
			continue
		}
		simplelog.Info("feed was successfully updated", "consumer", w.reloadHandlerSpecs[i].name, "hash", w.reloadHandlerSpecs[i].lastHash)
	}
}

// https://yadi.yandex-team.ru/db/{lang}.json.gz -> nodejs.json.gz
func parseFeedFilename(feedURL, lang string) string {
	feedPath := feed.FormatURI(feedURL, lang)
	parts := strings.Split(feedPath, "/")
	return parts[len(parts)-1]
}
