import json
from typing import List

import requests


class Service(dict):
    def get_contact_by_type(self, contact_type, default=False):
        # Ok, let's assume we get items from self
        for c in self.get("contacts", []):
            if c["contact_type"] == contact_type:
                return c["content"]
        return default


class Abc(object):
    base_url = ""
    token = ""

    def __init__(self, base_url, token):
        self.base_url = base_url
        self.token = token
        self.headers = {
            "content-type": "application/json",
            "Authorization": "OAuth " + self.token,
        }

    def get_service_by_slug(self, slug, fields=None):
        payload = {"slug": slug}
        if fields:
            payload["fields"] = fields
        url = self.base_url + "/services/"
        r = requests.get(url, params=payload, headers=self.headers)
        json_data = r.json()
        if json_data["results"]:
            return json_data["results"][0]
        return None

    def get_service_by_id(self, abc_id, fields=None):
        try:
            int(abc_id)
        except (ValueError, TypeError):
            print("abc_id must be a number, not %s" % abc_id)
        payload = {}
        if fields:
            payload["fields"] = fields
        url = "%s%s%s%s" % (self.base_url, "/services/", abc_id, "/")
        r = requests.get(url, headers=self.headers, params=payload)
        json_data = r.json()
        return json_data

    def get_service_children(self, abc_id: int, fields=None):
        url = f'{self.base_url}/services/?parent={int(abc_id)}'
        payload = dict()
        if fields:
            payload["fields"] = fields
        r = requests.get(url, headers=self.headers, params=payload)
        json_data = r.json()
        results = [*json_data.get('results')]
        next_page = json_data["next"]
        while next_page:
            results.extend(json_data.get('results'))
        return results

    def get_service_by_id_with_descendants(self, abc_id, fields=None):
        try:
            int(abc_id)
        except (ValueError, TypeError):
            print("abc_id must be a number, not %s" % abc_id)
        service = self.get_service_by_id(abc_id)
        result = [service]
        payload = {
            "page_size": 1000,
            "parent__with_descendants": abc_id,
        }
        if fields:
            payload["fields"] = fields
        next_page = self.base_url + "/services/"
        while next_page:
            r = requests.get(next_page, params=payload, headers=self.headers)
            json_data = r.json()
            if json_data and "error" not in json_data:
                result += json_data["results"]
                next_page = json_data["next"]
            else:
                next_page = None
        return result

    def get_service_responsibles(self, abc_id):
        try:
            int(abc_id)
        except (ValueError, TypeError):
            print("abc_id must be a number, not %s" % abc_id)
        payload = {"service": abc_id}
        url = self.base_url + "/services/responsibles/"
        r = requests.get(url, params=payload, headers=self.headers)
        if r.status_code != 200:
            return None
        json_data = r.json()
        responsibles = list()
        for person in json_data["results"]:
            responsibles.append(person["person"]["login"])
        return responsibles

    def get_service_head_by_id(self, abc_id):
        try:
            int(abc_id)
        except (ValueError, TypeError):
            print("abc_id must be a number, not %s" % abc_id)
        payload = {"service": abc_id, "role": 1}
        url = self.base_url + "/services/members/"
        r = requests.get(url, params=payload, headers=self.headers)
        if r.status_code != 200:
            return None
        json_data = r.json()
        head = None
        if json_data["results"]:
            head = json_data["results"][0]["person"]["login"]
        return head

    def get_service_by_startrek(self, queue_key, fields=None):
        result = {}
        payload = {
            "contact_type": 4,
            "content__contains": queue_key,
        }
        if fields:
            payload["fields"] = fields
        next_page = self.base_url + "/services/contacts/"
        while next_page:
            r = requests.get(next_page, params=payload, headers=self.headers)
            json_data = r.json()
            if json_data:
                next_page = json_data.get("next", None)
                for s in json_data["results"]:
                    if s["content"] in result:
                        result[s["content"]].append(s["service"]["id"])
                    else:
                        result[s["content"]] = [s["service"]["id"]]
        if queue_key in result:
            for abc_id in result[queue_key]:
                abc_record = self.get_service_by_id(abc_id)
                if abc_record["state"] not in {"deleted"}:
                    return abc_record
            if result[queue_key]:
                return self.get_service_by_id(result[queue_key][0])
        return {}

    def get_members_info_by_id(
        self,
        service_id: int,
        with_descendants: bool = False,
        use_inheritance_settings: bool = True,
        role_scope: str = None,
        fields=None,
    ):
        try:
            int(service_id)
        except (ValueError, TypeError):
            print("service_id must be a number, not %s" % service_id)
        url = self.base_url + "/services/members/"
        if with_descendants:
            payload = {"service__with_descendants": service_id}
        else:
            payload = {"service": service_id}
        if use_inheritance_settings:
            payload["use_inheritance_settings"] = "True"
        if role_scope:
            if isinstance(role_scope, str):
                role_scope = role_scope.split(",")
            payload["role__scope"] = role_scope
        if fields:
            payload["fields"] = fields
        response = requests.get(url, params=payload, headers=self.headers)
        parsed_json = response.json()
        result = parsed_json["results"]
        while True:  # pagging
            if "next" in parsed_json and parsed_json["next"]:
                url = parsed_json["next"]
                response = requests.get(url, params=payload, headers=self.headers)
                parsed_json = json.loads(response.content)
                result += parsed_json["results"]
            else:
                break
        return result

    def get_people_by_id(
        self,
        service_id: int,
        with_descendants: bool = True,
        use_inheritance_settings: bool = False,
        role_scope: str = None,
        role_codes: List = None,
        fields=None,
    ) -> List:
        """
        get people in service and subservices
        :param service_id:
        :type service_id: int
        :param with_descendants: fetch with subservices. Optional, defaults to *True*
        :type with_descendants: bool
        :param role_scope: filtered role scope. Optional, defaults to *None*
        :type role_scope: str
        :return:return set of logins
        """
        try:
            int(service_id)
        except (ValueError, TypeError):
            print("service_id must be a number, not %s" % service_id)
        url = self.base_url + "/services/members/"
        if with_descendants:
            payload = {"service__with_descendants": service_id}
        else:
            payload = {"service": service_id}
        if use_inheritance_settings:
            payload["use_inheritance_settings"] = "True"
        if role_scope:
            if isinstance(role_scope, str):
                role_scope = role_scope.split(",")
            payload["role__scope"] = role_scope
        if fields:
            payload["fields"] = fields
        response = requests.get(url, params=payload, headers=self.headers)
        parsed_json = response.json()
        result = parsed_json["results"]
        while True:  # pagging
            if "next" in parsed_json and parsed_json["next"]:
                url = parsed_json["next"]
                response = requests.get(url, params=payload, headers=self.headers)
                parsed_json = json.loads(response.content)
                result += parsed_json["results"]
            else:
                break

        logins = set()
        for person in result:
            if role_codes:
                if person["role"]["code"] in role_codes:
                    logins.add(person["person"]["login"])
            else:
                logins.add(person["person"]["login"])
        list_of_logins = list(logins)
        return list_of_logins

    def count_people_by_id(self, service_id):
        try:
            int(service_id)
        except (ValueError, TypeError):
            print("service_id must be a number, not %s" % service_id)
        logins = self.get_people_by_id(service_id=service_id)
        count = len(logins)
        return count

    def get_resourses_by_supplier(
        self, supplier_id, limit=1000, fields=None, state=None, type=None
    ):
        url = self.base_url + "/resources/consumers/"
        payload = {"supplier": supplier_id}
        if state:
            payload["state"] = state
        if type:
            payload["type"] = type
        if limit:
            payload["page_size"] = limit
        if fields:
            payload["fields"] = fields

        payload["format"] = "json"

        response = requests.get(url, params=payload, headers=self.headers)
        parsed_json = response.json()
        result = parsed_json["results"]

        while True:  # pagging
            if "next" in parsed_json and parsed_json["next"]:
                url = parsed_json["next"]
                response = requests.get(url, params=payload, headers=self.headers)
                parsed_json = response.json()
                result += parsed_json["results"]
            else:
                break
        return result

    def get_shifts(self, service_id):
        url = "%s/duty/on_duty/" % self.base_url
        parameters = {"service": service_id}
        r = requests.get(url, headers=self.headers, params=parameters)
        json_data = r.json()

        return json_data
