import requests
import json
import logging

try:
    from urlparse import urljoin
except ImportError:
    from urllib.parse import urljoin

log = logging.getLogger(__name__)


class AppQueuer(object):
    def __init__(self, base_url="https://appcheck.yandex-team.ru", token=""):
        self.base_url = base_url
        self.token = token

    def init_scan(self, url, st_ticket, repository):
        payload = json.dumps(
            {"URL": url, "ticket_id": st_ticket, "repository": repository}
        )

        log.debug("URL: {}, Payload: {}".format(url, payload))
        r = requests.post(urljoin(self.base_url, "/api/analyze_url"), data=payload)
        if r.status_code == 200:
            resp = r.json()
            return resp
        return False

    def init_repository_scan(self, repo_url, branch, st_ticket):
        params = {"url": repo_url, "branch": branch, "ticket": st_ticket}

        payload = json.dumps(params)
        url = urljoin(self.base_url, "/api/coverity/scan/")

        log.debug("URL: {}, Payload: {}".format(url, payload))
        r = requests.post(url, data=payload)

        if r.text:
            log.debug("Response: {}".format(r.text))
        if r.status_code != 200:
            return False

        return json.loads(r.text)["scan_id"]

    def init_package_scan(self, package_url, st_ticket):
        params = {"package": package_url, "ticket": st_ticket}

        payload = json.dumps(params)
        url = urljoin(self.base_url, "/api/repository/analyze/")

        log.debug("URL: {}, Payload: {}".format(url, payload))
        r = requests.post(url, data=payload)

        if r.status_code != 200:
            return False

        return json.loads(r.text)["scan_id"]

    def init_android_application_scan(self, beta_url, ticket_id, assignee, queue):
        params = {
            "url": beta_url,
            "assignee": assignee,
            "secaudit_ticket": ticket_id,
            "queue": queue,
        }
        url = urljoin(self.base_url, "/api/android/analyze/")
        payload = json.dumps(params)

        log.debug("URL: {}, Payload: {}".format(url, payload))
        r = requests.post(url, data=payload)

        if r.status_code != 200:
            return False

        return json.loads(r.text)["scan_id"]

    def init_yodax_repo_scan(self, repo_url, branch, st_ticket, assignee, queue):
        params = {
            "url": repo_url,
            "assignee": assignee,
            "ticket": st_ticket,
            "queue": queue,
            "branch": branch,
        }

        payload = json.dumps(params)
        url = urljoin(self.base_url, "/api/yodax/scan/")

        log.debug("URL: {}".format(url))
        log.debug("Payload: {}".format(payload))

        r = requests.post(url, json=params)

        if r.text:
            log.debug("Response: {}".format(r.text))

        if r.status_code != 200:
            return False

        return json.loads(r.text)["scan_id"]

    def get_scan_results(self, scan_id):
        url = urljoin(self.base_url, "/api/coverity/{}/".format(scan_id))
        log.debug("URL: {}".format(url))

        r = requests.get(url)

        if r.status_code != 200:
            return False

        return json.loads(r.text)

    def get_android_app_scan_status(self, scan_id):
        url = urljoin(self.base_url, "/api/android/{}/results/".format(scan_id))
        log.debug("URL: {}".format(url))

        r = requests.get(url)

        if r.status_code != 200:
            return False

        return json.loads(r.text)

    def init_bandit_scan(self, repo_url, branch, st_ticket):
        params = {"url": repo_url, "branch": branch, "ticket": st_ticket}

        url = urljoin(self.base_url, "/api/bandit/scan/")
        payload = json.dumps(params)

        log.debug("URL: {}, Payload: {}".format(url, payload))
        r = requests.post(url, data=payload)

        if r.text:
            log.debug("Response: {}".format(r.text))

        if r.status_code != 200:
            return False

        return json.loads(r.text)["scan_id"]

    def get_bandit_scan_results(self, scan_id):
        url = urljoin(self.base_url, "/api/bandit/{}/".format(scan_id))
        log.debug("URL: {}".format(url))

        r = requests.get(url)

        if r.status_code != 200:
            return False

        return json.loads(r.text)

    def init_secrets_scan(self, repo_url, branch, st_ticket):
        params = {"url": repo_url, "branch": branch, "ticket": st_ticket}
        url = urljoin(self.base_url, "/api/secrets/scan/")
        log.debug("URL: {}".format(url))
        log.debug("Payload: {}".format(json.dumps(params)))

        r = requests.post(url, json=params)

        if r.text:
            log.debug("Response: {}".format(r.text))

        if r.status_code != 200:
            return False

        return json.loads(r.text)["scan_id"]

    def init_yadi_scan(self, repo_url, branch, st_ticket):
        params = {"url": repo_url, "branch": branch, "ticket": st_ticket}

        url = urljoin(self.base_url, "/api/yadi/scan/")
        log.debug("URL: {}".format(url))
        log.debug("Payload: {}".format(json.dumps(params)))

        r = requests.post(url, json=params)
        if r.text:
            log.debug("Response: {}".format(r.text))

        if r.status_code != 200:
            return False

        return json.loads(r.text)["scan_id"]
