import logging
from functools import wraps
from typing import Callable

logger = logging.getLogger(__name__)


class DictVal:
    def __init__(self, val):
        self._val = val

    @staticmethod
    def __wrap(val):
        if isinstance(val, dict) or isinstance(val, list):
            return DictVal(val)
        return val

    def __getattr__(self, item: str):
        res = self._val.get(item)
        return DictVal.__wrap(res)

    def __getitem__(self, item):
        res = self._val[item]
        return DictVal.__wrap(res)

    def __str__(self):
        return str(self._val)

    def __int__(self):
        return int(self._val)

    def __float__(self):
        return float(self._val)

    def __iter__(self):
        return iter(map(lambda x: DictVal.__wrap(x), self._val))


def safe_func(func: Callable) -> Callable:
    """
    Decorator to make function safe in terms of throwing exceptions.
    """

    @wraps(func)
    def _wrapper(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except Exception as e:
            logger.exception(e)
            return None

    return _wrapper
