from requests import get
from time import strptime, mktime


class Dogma(object):
    base_url = ""
    token = ""

    def __init__(self, base_url, token):
        self.base_url = base_url
        self.token = token
        self.headers = {"Authorization": "OAuth " + self.token}

    def get_repositories_by_queue(self, queue, until_ts=0):
        """
        Get all repositories related to queue
        by reading all attached commits from dogma.
        If use with until_ts param - read only newer attached commits.
        """

        repos = set()
        page = 1
        running = True

        while running:
            url = (
                "{}/search/commits?"
                "q=queues:{}+"
                "exclude_dublicates:true+parents:false&numpages=true"
                "&per_page=200&page={}".format(self.base_url, queue, page)
            )
            r = get(url, headers=self.headers, verify=False)
            if r.status_code != 200:
                break
            commits = r.json()

            for commit in commits:
                if until_ts:
                    date_str = commit["commit"]["author"]["date"]
                    date = strptime(date_str, "%Y-%m-%dT%H:%M:%S")
                    ts = int(mktime(date))
                    if ts < until_ts:
                        running = False
                        break

                repo = commit.get("repo", {})
                url = repo.get("url")
                if url:
                    repos.add(url)

            if not r.headers.get("Link"):
                break
            page += 1

        return list(repos)

    def get_last_commits(self, queue, count=0):
        """
        Get last commits related to queue.
        If count=0 - get all commits.
        """

        commits = list()
        page = 1
        read_count = 0

        while True:
            if count:
                per_page = min(count - read_count, 200)
            else:
                per_page = 200
            url = (
                "{}/search/commits?"
                "q=queues:{}+"
                "exclude_dublicates:true+parents:false&numpages=true"
                "&per_page={}"
                "&page={}".format(self.base_url, queue, per_page, page)
            )
            r = get(url, headers=self.headers, verify=False)
            if r.status_code != 200:
                break
            json_data = r.json()
            read_count += len(json_data)
            commits += json_data

            if read_count == count or not r.headers.get("Link"):
                break
            page += 1

        return commits
